/**
 * WooCommerce Quotes for Woodmart - Frontend Scripts
 */

(function($) {
    'use strict';
    
    var WCQ_Frontend = {
        
        /**
         * Initialisation
         */
        init: function() {
            this.bindEvents();
        },
        
        /**
         * Lier les événements
         */
        bindEvents: function() {
            $(document).on('click', '.wcq-email-quote-btn', this.emailQuote);
            $(document).on('click', '.wcq-delete-quote-btn', this.deleteQuote);
            
            // Gestion des quantités
            $(document).on('change', '.wcq-quantity-input', this.updateItemTotal);
            $(document).on('click', '.wcq-qty-plus', this.increaseQuantity);
            $(document).on('click', '.wcq-qty-minus', this.decreaseQuantity);
            
            // Gestion de la suppression d'articles
            $(document).on('click', '.wcq-remove-item', this.removeItem);
            
            // Validation du formulaire
            $(document).on('submit', '.wcq-update-quote-form', this.validateUpdateForm);
        },
        
        /**
         * Augmenter la quantité (bouton +)
         */
        increaseQuantity: function(e) {
            e.preventDefault();
            var $input = $(this).siblings('.wcq-quantity-input');
            var currentVal = parseInt($input.val()) || 1;
            $input.val(currentVal + 1).trigger('change');
        },
        
        /**
         * Diminuer la quantité (bouton -)
         */
        decreaseQuantity: function(e) {
            e.preventDefault();
            var $input = $(this).siblings('.wcq-quantity-input');
            var currentVal = parseInt($input.val()) || 1;
            if (currentVal > 1) {
                $input.val(currentVal - 1).trigger('change');
            }
        },
        
        /**
         * Mettre à jour le total d'un article lors du changement de quantité
         */
        updateItemTotal: function() {
            var $input = $(this);
            var quantity = parseInt($input.val()) || 1;
            var unitPrice = parseFloat($input.data('unit-price')) || 0;
            var itemIndex = $input.data('item-index');
            
            // Calculer le nouveau total de l'article
            var itemTotal = quantity * unitPrice;
            
            // Mettre à jour l'affichage du total de l'article
            var $itemTotalSpan = $('.wcq-item-total[data-item-index="' + itemIndex + '"]');
            $itemTotalSpan.html(WCQ_Frontend.formatPrice(itemTotal));
            
            // Recalculer le total général
            WCQ_Frontend.updateGrandTotal();
        },
        
        /**
         * Recalculer le total général du devis
         */
        updateGrandTotal: function() {
            var grandTotal = 0;
            
            $('.wcq-quantity-input').each(function() {
                // Ignorer les champs marqués comme supprimés
                if ($(this).attr('data-removed') === '1') {
                    return;
                }
                
                var $row = $(this).closest('tr');
                
                // Vérifier si l'article n'est pas marqué pour suppression
                if ($row.find('.wcq-remove-flag').val() !== '1') {
                    var quantity = parseInt($(this).val()) || 0;
                    var unitPrice = parseFloat($(this).data('unit-price')) || 0;
                    grandTotal += quantity * unitPrice;
                }
            });
            
            // Mettre à jour l'affichage du total général
            $('.wcq-grand-total').html(WCQ_Frontend.formatPrice(grandTotal));
        },
        
        /**
         * Formater un prix avec les paramètres WooCommerce
         */
        formatPrice: function(price) {
            // Utiliser wc_price si disponible (paramètres passés par PHP)
            if (typeof wcq_params !== 'undefined' && wcq_params.currency_format) {
                var formatted = price.toFixed(wcq_params.decimals || 2);
                formatted = formatted.replace('.', wcq_params.decimal_sep || ',');
                
                // Ajouter les séparateurs de milliers
                if (wcq_params.thousand_sep) {
                    var parts = formatted.split(wcq_params.decimal_sep);
                    parts[0] = parts[0].replace(/\B(?=(\d{3})+(?!\d))/g, wcq_params.thousand_sep);
                    formatted = parts.join(wcq_params.decimal_sep);
                }
                
                // Appliquer le format (symbole avant ou après)
                return wcq_params.currency_format
                    .replace('%1$s', wcq_params.currency_symbol || '€')
                    .replace('%2$s', formatted);
            }
            
            // Fallback : format français par défaut
            var formatted = price.toFixed(2).replace('.', ',');
            return formatted + '\u00A0€'; // \u00A0 = espace insécable
        },
        
        /**
         * Supprimer un article du devis
         */
        removeItem: function(e) {
            e.preventDefault();
            
            var confirmText = (typeof wcq_params !== 'undefined' && wcq_params.i18n && wcq_params.i18n.confirm_remove) 
                ? wcq_params.i18n.confirm_remove 
                : 'Êtes-vous sûr de vouloir supprimer cet article du devis ?';
                
            if (!confirm(confirmText)) {
                return;
            }
            
            var $button = $(this);
            var $row = $button.closest('tr');
            var itemIndex = $button.data('item-index');
            
            // Marquer l'article pour suppression
            $row.find('.wcq-remove-flag').val('1');
            
            // Marquer le champ de quantité comme supprimé (mais ne pas le désactiver pour qu'il soit envoyé)
            var $quantityInput = $row.find('.wcq-quantity-input');
            $quantityInput.attr('data-removed', '1').val(0);
            
            // Masquer visuellement la ligne
            $row.addClass('wcq-item-removed').fadeOut(300);
            
            // Recalculer le total
            WCQ_Frontend.updateGrandTotal();
        },
        
        /**
         * Valider le formulaire de mise à jour
         */
        validateUpdateForm: function(e) {
            // Compter les articles non supprimés
            var remainingItems = 0;
            
            $('.wcq-quantity-input').each(function() {
                // Ne compter que les champs non marqués comme supprimés
                if ($(this).attr('data-removed') !== '1' && $(this).closest('tr').find('.wcq-remove-flag').val() !== '1') {
                    remainingItems++;
                }
            });
            
            if (remainingItems === 0) {
                e.preventDefault();
                alert('Le devis doit contenir au moins un article.');
                return false;
            }
            
            return true;
        },
        
        /**
         * Envoyer le devis par e-mail
         */
        emailQuote: function(e) {
            e.preventDefault();
            
            var $button = $(this);
            var quoteId = $button.data('quote-id');
            
            var email = prompt(wcq_params.i18n.loading || 'Entrez l\'adresse e-mail (laisser vide pour votre propre adresse) :');
            
            if (email === null) {
                return; // Annulé
            }
            
            $button.prop('disabled', true).addClass('wcq-loading');
            
            $.ajax({
                url: wcq_params.ajax_url,
                type: 'POST',
                data: {
                    action: 'wcq_send_quote_email',
                    nonce: wcq_params.nonce,
                    quote_id: quoteId,
                    email: email
                },
                success: function(response) {
                    if (response.success) {
                        WCQ_Frontend.showNotice(response.data.message, 'success');
                    } else {
                        WCQ_Frontend.showNotice(response.data.message || 'Erreur inconnue', 'error');
                    }
                    $button.prop('disabled', false).removeClass('wcq-loading');
                },
                error: function() {
                    WCQ_Frontend.showNotice('Erreur lors de l\'envoi de l\'e-mail', 'error');
                    $button.prop('disabled', false).removeClass('wcq-loading');
                }
            });
        },
        
        /**
         * Supprimer un devis
         */
        deleteQuote: function(e) {
            e.preventDefault();
            
            if (!confirm(wcq_params.i18n.confirm_delete || 'Êtes-vous sûr de vouloir supprimer ce devis ?')) {
                return;
            }
            
            var $button = $(this);
            var quoteId = $button.data('quote-id');
            
            $button.prop('disabled', true).addClass('wcq-loading');
            
            $.ajax({
                url: wcq_params.ajax_url,
                type: 'POST',
                data: {
                    action: 'wcq_delete_quote',
                    nonce: wcq_params.nonce,
                    quote_id: quoteId
                },
                success: function(response) {
                    if (response.success) {
                        WCQ_Frontend.showNotice(response.data.message, 'success');
                        
                        // Rediriger vers la liste des devis après 1 seconde
                        setTimeout(function() {
                            window.location.href = window.location.href.split('?')[0].replace(/\/\d+\/$/, '/');
                        }, 1000);
                    } else {
                        WCQ_Frontend.showNotice(response.data.message || 'Erreur inconnue', 'error');
                        $button.prop('disabled', false).removeClass('wcq-loading');
                    }
                },
                error: function() {
                    WCQ_Frontend.showNotice('Erreur lors de la suppression du devis', 'error');
                    $button.prop('disabled', false).removeClass('wcq-loading');
                }
            });
        },
        
        /**
         * Afficher une notification
         */
        showNotice: function(message, type) {
            type = type || 'info';
            
            var $notice = $('<div class="woocommerce-message wcq-notice wcq-notice-' + type + '">' + message + '</div>');
            
            $('.woocommerce-notices-wrapper').first().html($notice);
            
            $('html, body').animate({
                scrollTop: $notice.offset().top - 100
            }, 500);
            
            // Supprimer automatiquement après 5 secondes
            setTimeout(function() {
                $notice.fadeOut(function() {
                    $(this).remove();
                });
            }, 5000);
        }
    };
    
    // Initialiser au chargement du document
    $(document).ready(function() {
        WCQ_Frontend.init();
    });
    
})(jQuery);
