<?php
/**
 * Gestion de l'interface d'administration
 *
 * @package WC_Quotes_Woodmart
 */

if (!defined('ABSPATH')) {
    exit;
}

class WCQ_Admin {
    
    /**
     * Constructeur
     */
    public function __construct() {
        // Ajouter le menu dans WooCommerce
        add_action('admin_menu', array($this, 'add_admin_menu'), 60);

        // Personnaliser les colonnes de la liste des devis
        add_filter('manage_shop_quote_posts_columns', array($this, 'quote_columns'));
        add_action('manage_shop_quote_posts_custom_column', array($this, 'quote_column_content'), 10, 2);

        // Ajouter des filtres sur la liste des devis
        add_filter('views_edit-shop_quote', array($this, 'quote_views'));

        // Modifier la requête pour afficher tous les statuts dans l'onglet "Tous"
        add_filter('pre_get_posts', array($this, 'modify_quote_query'));

        // Ajouter les meta boxes
        add_action('add_meta_boxes', array($this, 'add_meta_boxes'));

        // Sauvegarder les meta boxes
        add_action('save_post_shop_quote', array($this, 'save_meta_boxes'), 10, 2);

        // Supprimer les éléments de menu inutiles
        add_action('admin_head', array($this, 'remove_quick_edit'));

        // Afficher les messages admin personnalisés
        add_action('admin_notices', array($this, 'customer_change_notice'));
    }
    
    /**
     * Ajouter le menu dans WooCommerce
     */
    public function add_admin_menu() {
        add_submenu_page(
            'woocommerce',
            __('Devis', 'wc-quotes-woodmart'),
            __('Devis', 'wc-quotes-woodmart'),
            'manage_woocommerce',
            'edit.php?post_type=shop_quote'
        );
    }
    
    /**
     * Personnaliser les colonnes de la liste des devis
     */
    public function quote_columns($columns) {
        $new_columns = array();
        
        $new_columns['cb'] = $columns['cb'];
        $new_columns['quote_number'] = __('Numéro', 'wc-quotes-woodmart');
        $new_columns['customer'] = __('Client', 'wc-quotes-woodmart');
        $new_columns['total'] = __('Total', 'wc-quotes-woodmart');
        $new_columns['status'] = __('Statut', 'wc-quotes-woodmart');
        $new_columns['valid_until'] = __('Valide jusqu\'au', 'wc-quotes-woodmart');
        $new_columns['date'] = __('Date', 'wc-quotes-woodmart');
        $new_columns['actions'] = __('Actions', 'wc-quotes-woodmart');
        
        return $new_columns;
    }
    
    /**
     * Contenu des colonnes personnalisées
     */
    public function quote_column_content($column, $post_id) {
        $quote = WCQ_Quote::get_quote($post_id);
        
        switch ($column) {
            case 'quote_number':
                echo '<strong><a href="' . get_edit_post_link($post_id) . '">#' . esc_html($quote['quote_number']) . '</a></strong>';
                break;
                
            case 'customer':
                $customer_id = $quote['customer_id'];
                if ($customer_id > 0) {
                    $user = get_userdata($customer_id);
                    if ($user) {
                        echo '<a href="' . admin_url('user-edit.php?user_id=' . $customer_id) . '">';
                        echo esc_html($user->display_name);
                        echo '</a><br>';
                        echo '<small>' . esc_html($user->user_email) . '</small>';

                        // Si le devis a été transféré, afficher le nouveau client
                        if ($quote['status'] === 'quote-transferred') {
                            $transferred_to_id = get_post_meta($post_id, '_transferred_to', true);
                            if ($transferred_to_id) {
                                $new_user = get_userdata($transferred_to_id);
                                if ($new_user) {
                                    echo '<br><small style="color: #2271b1; font-weight: 600;">';
                                    echo '→ Transféré à: ';
                                    echo '<a href="' . admin_url('user-edit.php?user_id=' . $transferred_to_id) . '">';
                                    echo esc_html($new_user->display_name);
                                    echo '</a>';
                                    echo '</small>';
                                }
                            }
                        }
                    }
                } else {
                    echo '—';
                }
                break;
                
            case 'total':
                echo wc_price($quote['total']);
                break;
                
            case 'status':
                $status_name = $this->get_status_name($quote['status']);
                $status_class = str_replace('quote-', '', $quote['status']);
                echo '<span class="wcq-status wcq-status-' . esc_attr($status_class) . '">' . esc_html($status_name) . '</span>';
                break;
                
            case 'valid_until':
                if ($quote['valid_until']) {
                    $valid_date = strtotime($quote['valid_until']);
                    $now = current_time('timestamp');
                    
                    if ($valid_date < $now) {
                        echo '<span style="color: red;">' . date_i18n(get_option('date_format'), $valid_date) . '</span>';
                    } else {
                        echo date_i18n(get_option('date_format'), $valid_date);
                    }
                } else {
                    echo '—';
                }
                break;
                
            case 'actions':
                ?>
                <button type="button" class="button button-small wcq-convert-quote" data-quote-id="<?php echo esc_attr($post_id); ?>">
                    <?php _e('Convertir en commande', 'wc-quotes-woodmart'); ?>
                </button>
                <?php
                break;
        }
    }
    
    /**
     * Obtenir le nom du statut
     */
    private function get_status_name($status) {
        $statuses = array(
            'quote-pending'      => __('En attente', 'wc-quotes-woodmart'),
            'quote-accepted'     => __('Accepté', 'wc-quotes-woodmart'),
            'quote-rejected'     => __('Refusé', 'wc-quotes-woodmart'),
            'quote-expired'      => __('Expiré', 'wc-quotes-woodmart'),
            'quote-transferred'  => __('Transféré', 'wc-quotes-woodmart'),
        );

        return isset($statuses[$status]) ? $statuses[$status] : $status;
    }
    
    /**
     * Ajouter les filtres de vue
     */
    public function quote_views($views) {
        // Compter les devis par statut
        $statuses = array(
            'all' => array(
                'label' => __('Tous', 'wc-quotes-woodmart'),
                'count' => 0
            ),
            'quote-pending' => array(
                'label' => __('En attente', 'wc-quotes-woodmart'),
                'count' => 0
            ),
            'quote-accepted' => array(
                'label' => __('Accepté', 'wc-quotes-woodmart'),
                'count' => 0
            ),
            'quote-rejected' => array(
                'label' => __('Refusé', 'wc-quotes-woodmart'),
                'count' => 0
            ),
            'quote-expired' => array(
                'label' => __('Expiré', 'wc-quotes-woodmart'),
                'count' => 0
            ),
            'quote-replaced' => array(
                'label' => __('Remplacé', 'wc-quotes-woodmart'),
                'count' => 0
            ),
            'quote-transferred' => array(
                'label' => __('Transféré', 'wc-quotes-woodmart'),
                'count' => 0
            ),
        );

        // Compter les devis pour chaque statut
        global $wpdb;
        $counts = $wpdb->get_results(
            "SELECT post_status, COUNT(*) as count
            FROM {$wpdb->posts}
            WHERE post_type = 'shop_quote'
            GROUP BY post_status",
            ARRAY_A
        );

        $total = 0;
        foreach ($counts as $count) {
            $status = $count['post_status'];
            $num = (int) $count['count'];
            $total += $num;

            if (isset($statuses[$status])) {
                $statuses[$status]['count'] = $num;
            }
        }
        $statuses['all']['count'] = $total;

        // Construire les liens de filtre
        $current_status = isset($_GET['post_status']) ? $_GET['post_status'] : 'all';
        $base_url = admin_url('edit.php?post_type=shop_quote');

        $new_views = array();

        // Lien "Tous"
        $class = ($current_status === 'all' || empty($current_status)) ? 'current' : '';
        $new_views['all'] = sprintf(
            '<a href="%s" class="%s">%s <span class="count">(%d)</span></a>',
            $base_url,
            $class,
            $statuses['all']['label'],
            $statuses['all']['count']
        );

        // Liens pour chaque statut
        foreach ($statuses as $status => $data) {
            if ($status === 'all' || $data['count'] === 0) {
                continue;
            }

            $class = ($current_status === $status) ? 'current' : '';
            $new_views[$status] = sprintf(
                '<a href="%s" class="%s">%s <span class="count">(%d)</span></a>',
                add_query_arg('post_status', $status, $base_url),
                $class,
                $data['label'],
                $data['count']
            );
        }

        return $new_views;
    }

    /**
     * Modifier la requête pour afficher tous les statuts de devis dans "Tous"
     */
    public function modify_quote_query($query) {
        // Vérifier qu'on est dans l'admin et sur la page des devis
        if (!is_admin() || !$query->is_main_query()) {
            return;
        }

        // Vérifier qu'on est sur la page des devis
        $screen = get_current_screen();
        if (!$screen || $screen->post_type !== 'shop_quote') {
            return;
        }

        // Si aucun statut n'est spécifié (onglet "Tous"), inclure tous les statuts de devis
        if (!isset($_GET['post_status'])) {
            $query->set('post_status', array(
                'quote-pending',
                'quote-accepted',
                'quote-rejected',
                'quote-expired',
                'quote-replaced',
                'quote-transferred'
            ));
        }
    }

    /**
     * Ajouter les meta boxes
     */
    public function add_meta_boxes() {
        add_meta_box(
            'wcq_quote_details',
            __('Détails du devis', 'wc-quotes-woodmart'),
            array($this, 'quote_details_meta_box'),
            'shop_quote',
            'normal',
            'high'
        );
        
        add_meta_box(
            'wcq_quote_actions',
            __('Actions', 'wc-quotes-woodmart'),
            array($this, 'quote_actions_meta_box'),
            'shop_quote',
            'side',
            'high'
        );
        
        add_meta_box(
            'wcq_quote_customer',
            __('Informations client', 'wc-quotes-woodmart'),
            array($this, 'quote_customer_meta_box'),
            'shop_quote',
            'side',
            'default'
        );

        add_meta_box(
            'wcq_customer_change_log',
            __('Historique des modifications client', 'wc-quotes-woodmart'),
            array($this, 'customer_change_log_meta_box'),
            'shop_quote',
            'normal',
            'low'
        );
    }
    
    /**
     * Meta box : Détails du devis
     */
    public function quote_details_meta_box($post) {
        $quote = WCQ_Quote::get_quote($post->ID);
        $quote_data = $quote['quote_data'];
        
        if (empty($quote_data)) {
            echo '<p>' . __('Aucun produit dans ce devis.', 'wc-quotes-woodmart') . '</p>';
            return;
        }
        
        ?>
        <div class="wcq-quote-items">
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th><?php _e('Produit', 'wc-quotes-woodmart'); ?></th>
                        <th><?php _e('Prix unitaire', 'wc-quotes-woodmart'); ?></th>
                        <th><?php _e('Quantité', 'wc-quotes-woodmart'); ?></th>
                        <th><?php _e('Remise', 'wc-quotes-woodmart'); ?></th>
                        <th><?php _e('Total', 'wc-quotes-woodmart'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($quote_data as $item) :
                        $product_id = isset($item['variation_id']) && $item['variation_id'] > 0 ? $item['variation_id'] : $item['product_id'];
                        $product = wc_get_product($product_id);

                        if (!$product) continue;

                        // Calculer la remise à partir de line_subtotal et line_total
                        $line_subtotal = isset($item['line_subtotal']) ? floatval($item['line_subtotal']) : floatval($item['line_total']);
                        $line_total = floatval($item['line_total']);
                        $discount_percent = 0;
                        $has_discount = false;

                        if ($line_subtotal > $line_total && $line_subtotal > 0) {
                            $discount_percent = round((($line_subtotal - $line_total) / $line_subtotal) * 100);
                            $has_discount = true;
                        }

                        $unit_price_before = $line_subtotal / $item['quantity'];
                        $unit_price_after = $line_total / $item['quantity'];
                    ?>
                    <tr>
                        <td>
                            <?php echo $product->get_name(); ?>
                            <?php if (isset($item['variation']) && !empty($item['variation'])) : ?>
                                <div class="wcq-variation">
                                    <?php foreach ($item['variation'] as $key => $value) : ?>
                                        <small><?php echo wc_attribute_label($key) . ': ' . $value; ?></small><br>
                                    <?php endforeach; ?>
                                </div>
                            <?php endif; ?>
                            <?php
                            // Afficher les métadonnées personnalisées (comme la date de livraison)
                            $delivery_found = false;

                            // Parcourir toutes les clés de l'item pour trouver les dates de livraison
                            foreach ($item as $key => $value) {
                                // Exclure les clés standards
                                if (in_array($key, array('product_id', 'variation_id', 'variation', 'quantity', 'line_total', 'line_subtotal', 'line_tax', 'line_subtotal_tax', 'data', 'data_hash', 'key'))) {
                                    continue;
                                }

                                // Vérifier si c'est une date de livraison
                                $is_delivery = (stripos($key, 'livraison') !== false || stripos($key, 'delivery') !== false);

                                if ($is_delivery && !empty($value)) {
                                    echo '<div class="wcq-delivery-info" style="margin-top: 5px;">';
                                    echo '<small style="color: #2ea2cc;"><strong>Livraison estimée:</strong> ';

                                    // Si c'est un tableau ou un objet, extraire la valeur
                                    if (is_array($value)) {
                                        if (isset($value['value'])) {
                                            echo wp_kses_post($value['value']);
                                        } else {
                                            echo wp_kses_post(print_r($value, true));
                                        }
                                    } elseif (is_object($value)) {
                                        echo wp_kses_post(print_r($value, true));
                                    } else {
                                        echo wp_kses_post($value);
                                    }

                                    echo '</small></div>';
                                    $delivery_found = true;
                                    break;
                                }
                            }
                            ?>
                        </td>
                        <td>
                            <?php if ($has_discount) : ?>
                                <del><?php echo wc_price($unit_price_before); ?></del><br>
                                <ins><?php echo wc_price($unit_price_after); ?></ins>
                            <?php else : ?>
                                <?php echo wc_price($unit_price_after); ?>
                            <?php endif; ?>
                        </td>
                        <td><?php echo $item['quantity']; ?></td>
                        <td><?php echo $discount_percent > 0 ? $discount_percent . '%' : '—'; ?></td>
                        <td>
                            <?php if ($has_discount) : ?>
                                <del><?php echo wc_price($line_subtotal); ?></del><br>
                                <ins><?php echo wc_price($line_total); ?></ins>
                            <?php else : ?>
                                <?php echo wc_price($line_total); ?>
                            <?php endif; ?>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
                <tfoot>
                    <?php
                    $subtotal = 0;
                    foreach ($quote_data as $item) {
                        $subtotal += $item['line_total'];
                    }
                    $shipping = !empty($quote['shipping_cost']) ? $quote['shipping_cost'] : 0;
                    ?>
                    <tr>
                        <th colspan="4" style="text-align: right;"><?php _e('Sous-total', 'wc-quotes-woodmart'); ?></th>
                        <th><?php echo wc_price($subtotal); ?></th>
                    </tr>
                    <?php if ($shipping > 0) : ?>
                    <tr>
                        <th colspan="4" style="text-align: right;"><?php _e('Frais de port', 'wc-quotes-woodmart'); ?></th>
                        <th><?php echo wc_price($shipping); ?></th>
                    </tr>
                    <?php endif; ?>
                    <tr>
                        <th colspan="4" style="text-align: right; font-size: 14px;"><?php _e('Total TTC', 'wc-quotes-woodmart'); ?></th>
                        <th style="font-size: 14px;"><?php echo wc_price($quote['total']); ?></th>
                    </tr>
                </tfoot>
            </table>
        </div>
        <?php
    }
    
    /**
     * Meta box : Actions
     */
    public function quote_actions_meta_box($post) {
        $quote = WCQ_Quote::get_quote($post->ID);
        $converted_order_id = get_post_meta($post->ID, '_converted_order_id', true);
        
        ?>
        <div class="wcq-actions">
            <?php if ($converted_order_id) : ?>
                <p>
                    <?php 
                    printf(
                        __('Ce devis a été converti en commande #%s', 'wc-quotes-woodmart'),
                        '<a href="' . admin_url('post.php?post=' . $converted_order_id . '&action=edit') . '">' . $converted_order_id . '</a>'
                    ); 
                    ?>
                </p>
            <?php else : ?>
                <button type="button" class="button button-primary button-large wcq-convert-quote" data-quote-id="<?php echo esc_attr($post->ID); ?>" style="width: 100%; margin-bottom: 10px;">
                    <?php _e('Convertir en commande', 'wc-quotes-woodmart'); ?>
                </button>
            <?php endif; ?>
            
            <a href="<?php echo wp_nonce_url(admin_url('admin-post.php?action=wcq_download_pdf&quote_id=' . $post->ID), 'wcq_download_pdf'); ?>" class="button button-large" style="width: 100%; text-align: center; margin-bottom: 10px;">
                <?php _e('Télécharger en PDF', 'wc-quotes-woodmart'); ?>
            </a>
            
            <button type="button" class="button button-large wcq-email-quote" data-quote-id="<?php echo esc_attr($post->ID); ?>" style="width: 100%;">
                <?php _e('Envoyer par e-mail', 'wc-quotes-woodmart'); ?>
            </button>
        </div>
        
        <div class="wcq-status-selector" style="margin-top: 20px;">
            <label>
                <strong><?php _e('Statut du devis', 'wc-quotes-woodmart'); ?></strong>
            </label>
            <select name="wcq_quote_status" style="width: 100%; margin-top: 5px;">
                <option value="quote-pending" <?php selected($quote['status'], 'quote-pending'); ?>><?php _e('En attente', 'wc-quotes-woodmart'); ?></option>
                <option value="quote-accepted" <?php selected($quote['status'], 'quote-accepted'); ?>><?php _e('Accepté', 'wc-quotes-woodmart'); ?></option>
                <option value="quote-rejected" <?php selected($quote['status'], 'quote-rejected'); ?>><?php _e('Refusé', 'wc-quotes-woodmart'); ?></option>
                <option value="quote-expired" <?php selected($quote['status'], 'quote-expired'); ?>><?php _e('Expiré', 'wc-quotes-woodmart'); ?></option>
                <option value="quote-transferred" <?php selected($quote['status'], 'quote-transferred'); ?>><?php _e('Transféré', 'wc-quotes-woodmart'); ?></option>
            </select>
        </div>
        
        <div class="wcq-valid-until" style="margin-top: 20px;">
            <label>
                <strong><?php _e('Valide jusqu\'au', 'wc-quotes-woodmart'); ?></strong>
            </label>
            <input type="date" name="wcq_valid_until" value="<?php echo $quote['valid_until'] ? date('Y-m-d', strtotime($quote['valid_until'])) : ''; ?>" style="width: 100%; margin-top: 5px;">
        </div>

        <div class="wcq-shipping-cost" style="margin-top: 20px;">
            <label>
                <strong><?php _e('Frais de port', 'wc-quotes-woodmart'); ?></strong>
            </label>
            <input type="number" name="wcq_shipping_cost" value="<?php echo !empty($quote['shipping_cost']) ? esc_attr($quote['shipping_cost']) : ''; ?>" step="0.01" min="0" placeholder="0.00" style="width: 100%; margin-top: 5px;">
            <p class="description" style="margin-top: 5px; font-size: 11px;">
                <?php _e('Montant en euros des frais de port', 'wc-quotes-woodmart'); ?>
            </p>
        </div>
        <?php
    }
    
    /**
     * Meta box : Informations client
     */
    public function quote_customer_meta_box($post) {
        $customer_id = get_post_meta($post->ID, '_customer_id', true);

        // Ajouter un nonce pour la sécurité
        wp_nonce_field('wcq_change_customer', 'wcq_change_customer_nonce');

        // Récupérer tous les clients
        $customers = get_users(array(
            'role__in' => array('customer', 'administrator', 'shop_manager'),
            'orderby'  => 'display_name',
            'order'    => 'ASC',
        ));

        ?>
        <div class="wcq-customer-selector" style="margin-bottom: 20px; padding: 15px; background: #f9f9f9; border: 1px solid #ddd; border-radius: 4px;">
            <label for="wcq_customer_id">
                <strong><?php _e('Attribuer à un client', 'wc-quotes-woodmart'); ?></strong>
            </label>
            <select name="wcq_customer_id" id="wcq_customer_id" style="width: 100%; margin-top: 5px; padding: 5px;">
                <option value=""><?php _e('-- Sélectionner un client --', 'wc-quotes-woodmart'); ?></option>
                <?php foreach ($customers as $customer) : ?>
                    <option value="<?php echo esc_attr($customer->ID); ?>" <?php selected($customer_id, $customer->ID); ?>>
                        <?php echo esc_html($customer->display_name); ?> (<?php echo esc_html($customer->user_email); ?>)
                    </option>
                <?php endforeach; ?>
            </select>
            <p class="description" style="margin-top: 8px;">
                <?php _e('Sélectionnez un client pour lui attribuer ce devis. Le changement sera enregistré lors de la mise à jour du devis.', 'wc-quotes-woodmart'); ?>
            </p>
        </div>

        <?php if ($customer_id > 0) : ?>
            <?php
            $user = get_userdata($customer_id);
            if ($user) :
            ?>
                <div class="wcq-customer-info">
                    <h4 style="margin-top: 0; padding-top: 0; border-top: 1px solid #ddd; padding-top: 15px;">
                        <?php _e('Informations du client actuel', 'wc-quotes-woodmart'); ?>
                    </h4>

                    <p><strong><?php _e('Nom', 'wc-quotes-woodmart'); ?>:</strong><br>
                    <?php echo esc_html($user->display_name); ?></p>

                    <p><strong><?php _e('E-mail', 'wc-quotes-woodmart'); ?>:</strong><br>
                    <a href="mailto:<?php echo esc_attr($user->user_email); ?>"><?php echo esc_html($user->user_email); ?></a></p>

                    <p><strong><?php _e('Téléphone', 'wc-quotes-woodmart'); ?>:</strong><br>
                    <?php
                    $phone = get_user_meta($customer_id, 'billing_phone', true);
                    echo $phone ? esc_html($phone) : '—';
                    ?></p>

                    <p>
                        <a href="<?php echo admin_url('user-edit.php?user_id=' . $customer_id); ?>" class="button">
                            <?php _e('Voir le profil', 'wc-quotes-woodmart'); ?>
                        </a>
                    </p>
                </div>
            <?php endif; ?>
        <?php else : ?>
            <p><?php _e('Aucun client associé', 'wc-quotes-woodmart'); ?></p>
        <?php endif; ?>
        <?php
    }
    
    /**
     * Sauvegarder les meta boxes
     */
    public function save_meta_boxes($post_id, $post) {
        // Vérifier les permissions
        if (!current_user_can('edit_post', $post_id)) {
            return;
        }

        // Éviter la boucle infinie
        remove_action('save_post_shop_quote', array($this, 'save_meta_boxes'), 10);

        // Sauvegarder le statut
        if (isset($_POST['wcq_quote_status'])) {
            wp_update_post(array(
                'ID' => $post_id,
                'post_status' => sanitize_text_field($_POST['wcq_quote_status'])
            ));
        }

        // Sauvegarder la date de validité
        if (isset($_POST['wcq_valid_until'])) {
            $valid_until = sanitize_text_field($_POST['wcq_valid_until']);
            if (!empty($valid_until)) {
                update_post_meta($post_id, '_valid_until', $valid_until . ' 23:59:59');
            } else {
                delete_post_meta($post_id, '_valid_until');
            }
        }

        // Sauvegarder les frais de port
        if (isset($_POST['wcq_shipping_cost'])) {
            $shipping_cost = floatval($_POST['wcq_shipping_cost']);
            if ($shipping_cost > 0) {
                update_post_meta($post_id, '_shipping_cost', $shipping_cost);
            } else {
                delete_post_meta($post_id, '_shipping_cost');
            }

            // Recalculer le total du devis en incluant les frais de port
            $quote_data = get_post_meta($post_id, '_quote_data', true);
            $subtotal = 0;
            if (is_array($quote_data)) {
                foreach ($quote_data as $item) {
                    $subtotal += isset($item['line_total']) ? $item['line_total'] : 0;
                }
            }
            $new_total = $subtotal + $shipping_cost;
            update_post_meta($post_id, '_quote_total', $new_total);
        }

        // Gérer le transfert de devis vers un autre client
        if (isset($_POST['wcq_customer_id']) && isset($_POST['wcq_change_customer_nonce'])) {
            // Vérifier le nonce
            if (!wp_verify_nonce($_POST['wcq_change_customer_nonce'], 'wcq_change_customer')) {
                add_action('save_post_shop_quote', array($this, 'save_meta_boxes'), 10, 2);
                return;
            }

            $new_customer_id = intval($_POST['wcq_customer_id']);
            $old_customer_id = get_post_meta($post_id, '_customer_id', true);

            // Si le client a changé et que le nouveau client est valide
            if ($new_customer_id > 0 && $new_customer_id !== intval($old_customer_id)) {
                $new_user = get_userdata($new_customer_id);
                $old_user = get_userdata($old_customer_id);

                if ($new_user && $old_user) {
                    // Dupliquer le devis pour le nouveau client
                    $original_quote_number = get_post_meta($post_id, '_quote_number', true);
                    $new_quote_id = $this->duplicate_quote_for_transfer($post_id, $new_customer_id, $old_customer_id);

                    if ($new_quote_id) {
                        // Le devis original garde son numéro et reste chez l'ancien client
                        // Mais on change son statut en "transféré" pour indiquer qu'il a été transféré
                        wp_update_post(array(
                            'ID' => $post_id,
                            'post_status' => 'quote-transferred',
                        ), false, false);

                        // Enregistrer les métadonnées de transfert
                        update_post_meta($post_id, '_transferred_to', $new_customer_id);
                        update_post_meta($post_id, '_transferred_date', current_time('mysql'));
                        update_post_meta($post_id, '_transferred_by', get_current_user_id());
                        update_post_meta($post_id, '_new_quote_id', $new_quote_id);

                        // Enregistrer les informations de transfert dans le devis original
                        update_post_meta($post_id, '_transfer_from_name', $old_user->display_name);
                        update_post_meta($post_id, '_transfer_to_name', $new_user->display_name);

                        // Enregistrer dans le journal de modifications
                        $this->log_customer_change($post_id, $old_customer_id, $new_customer_id);

                        // Ajouter un message de succès dans l'admin
                        add_filter('redirect_post_location', function($location) use ($new_user, $original_quote_number) {
                            return add_query_arg(array(
                                'message' => 1,
                                'wcq_quote_transferred' => 1,
                                'wcq_new_customer_name' => urlencode($new_user->display_name),
                                'wcq_original_quote_number' => urlencode($original_quote_number)
                            ), $location);
                        });
                    }
                }
            }
        }

        // Réactiver le hook
        add_action('save_post_shop_quote', array($this, 'save_meta_boxes'), 10, 2);
    }

    /**
     * Transférer un devis à un nouveau client (méthode publique réutilisable)
     *
     * @param int $quote_id ID du devis à transférer
     * @param int $new_customer_id ID du nouveau client
     * @return int|false ID du nouveau devis créé ou false en cas d'erreur
     */
    public function transfer_quote($quote_id, $new_customer_id) {
        $old_customer_id = get_post_meta($quote_id, '_customer_id', true);

        // Vérifier que les utilisateurs existent
        $new_user = get_userdata($new_customer_id);
        $old_user = get_userdata($old_customer_id);

        if (!$new_user || !$old_user) {
            return false;
        }

        // Dupliquer le devis pour le nouveau client
        $original_quote_number = get_post_meta($quote_id, '_quote_number', true);
        $new_quote_id = $this->duplicate_quote_for_transfer($quote_id, $new_customer_id, $old_customer_id);

        if (!$new_quote_id) {
            return false;
        }

        // Mettre à jour le devis original
        wp_update_post(array(
            'ID' => $quote_id,
            'post_status' => 'quote-transferred',
        ), false, false);

        // Enregistrer les métadonnées de transfert
        update_post_meta($quote_id, '_transferred_to', $new_customer_id);
        update_post_meta($quote_id, '_transferred_date', current_time('mysql'));
        update_post_meta($quote_id, '_transferred_by', get_current_user_id());
        update_post_meta($quote_id, '_new_quote_id', $new_quote_id);
        update_post_meta($quote_id, '_transfer_from_name', $old_user->display_name);
        update_post_meta($quote_id, '_transfer_to_name', $new_user->display_name);

        // Enregistrer dans le journal de modifications
        $this->log_customer_change($quote_id, $old_customer_id, $new_customer_id);

        return $new_quote_id;
    }

    /**
     * Dupliquer un devis pour le transférer à un nouveau client
     */
    private function duplicate_quote_for_transfer($original_quote_id, $new_customer_id, $old_customer_id) {
        // Récupérer le devis original
        $original_quote = get_post($original_quote_id);
        if (!$original_quote) {
            return false;
        }

        // Récupérer les métadonnées du devis original
        $quote_data = get_post_meta($original_quote_id, '_quote_data', true);
        $quote_total = get_post_meta($original_quote_id, '_quote_total', true);
        $quote_number = get_post_meta($original_quote_id, '_quote_number', true);
        $valid_until = get_post_meta($original_quote_id, '_valid_until', true);

        $new_user = get_userdata($new_customer_id);
        if (!$new_user) {
            return false;
        }

        // Créer le nouveau devis avec le suffixe "-trans" (c'est celui qui va au nouveau client)
        $new_quote_id = wp_insert_post(array(
            'post_type'   => 'shop_quote',
            'post_status' => 'quote-pending',
            'post_title'  => sprintf(__('Devis #%s', 'wc-quotes-woodmart'), $quote_number . '-trans'),
            'post_author' => $new_customer_id,
        ));

        if (is_wp_error($new_quote_id) || !$new_quote_id) {
            return false;
        }

        // Copier toutes les métadonnées importantes avec le suffixe "-trans"
        update_post_meta($new_quote_id, '_quote_number', $quote_number . '-trans');
        update_post_meta($new_quote_id, '_quote_data', $quote_data);
        update_post_meta($new_quote_id, '_quote_total', $quote_total);
        update_post_meta($new_quote_id, '_valid_until', $valid_until);
        update_post_meta($new_quote_id, '_customer_id', $new_customer_id);
        update_post_meta($new_quote_id, '_is_read', '0'); // Marquer comme non lu par défaut

        // Sauvegarder les informations du nouveau client
        update_post_meta($new_quote_id, '_customer_email', $new_user->user_email);
        update_post_meta($new_quote_id, '_customer_first_name', get_user_meta($new_customer_id, 'first_name', true));
        update_post_meta($new_quote_id, '_customer_last_name', get_user_meta($new_customer_id, 'last_name', true));

        // Enregistrer l'information de transfert dans le nouveau devis
        update_post_meta($new_quote_id, '_transferred_from', $old_customer_id);
        update_post_meta($new_quote_id, '_transferred_from_quote', $original_quote_id);
        update_post_meta($new_quote_id, '_is_transferred_quote', true);
        update_post_meta($new_quote_id, '_transferred_by', get_current_user_id());

        // Récupérer les noms pour les stocker
        $old_user = get_userdata($old_customer_id);
        if ($old_user) {
            update_post_meta($new_quote_id, '_transfer_from_name', $old_user->display_name);
        }
        update_post_meta($new_quote_id, '_transfer_to_name', $new_user->display_name);

        return $new_quote_id;
    }

    /**
     * Meta box : Journal des modifications client
     */
    public function customer_change_log_meta_box($post) {
        $change_log = get_post_meta($post->ID, '_customer_change_log', true);

        if (empty($change_log) || !is_array($change_log)) {
            echo '<p>' . __('Aucune modification du client enregistrée.', 'wc-quotes-woodmart') . '</p>';
            return;
        }

        ?>
        <div class="wcq-change-log">
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th><?php _e('Date', 'wc-quotes-woodmart'); ?></th>
                        <th><?php _e('Ancien client', 'wc-quotes-woodmart'); ?></th>
                        <th><?php _e('Nouveau client', 'wc-quotes-woodmart'); ?></th>
                        <th><?php _e('Modifié par', 'wc-quotes-woodmart'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach (array_reverse($change_log) as $entry) : ?>
                        <tr>
                            <td>
                                <?php echo date_i18n(get_option('date_format') . ' ' . get_option('time_format'), $entry['timestamp']); ?>
                            </td>
                            <td>
                                <?php if ($entry['old_customer_id'] > 0) : ?>
                                    <a href="<?php echo admin_url('user-edit.php?user_id=' . $entry['old_customer_id']); ?>">
                                        <?php echo esc_html($entry['old_customer_name']); ?>
                                    </a>
                                    <br><small><?php echo esc_html($entry['old_customer_email']); ?></small>
                                <?php else : ?>
                                    <em><?php echo esc_html($entry['old_customer_name']); ?></em>
                                <?php endif; ?>
                            </td>
                            <td>
                                <a href="<?php echo admin_url('user-edit.php?user_id=' . $entry['new_customer_id']); ?>">
                                    <?php echo esc_html($entry['new_customer_name']); ?>
                                </a>
                                <br><small><?php echo esc_html($entry['new_customer_email']); ?></small>
                            </td>
                            <td>
                                <a href="<?php echo admin_url('user-edit.php?user_id=' . $entry['changed_by']); ?>">
                                    <?php echo esc_html($entry['changed_by_name']); ?>
                                </a>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <?php
    }

    /**
     * Enregistrer le changement de client dans le journal des modifications
     */
    private function log_customer_change($quote_id, $old_customer_id, $new_customer_id) {
        // Récupérer le journal existant
        $change_log = get_post_meta($quote_id, '_customer_change_log', true);
        if (!is_array($change_log)) {
            $change_log = array();
        }

        // Obtenir les informations des clients
        $old_user = $old_customer_id > 0 ? get_userdata($old_customer_id) : null;
        $new_user = get_userdata($new_customer_id);
        $current_user = wp_get_current_user();

        // Ajouter l'entrée au journal
        $change_log[] = array(
            'date' => current_time('mysql'),
            'timestamp' => current_time('timestamp'),
            'old_customer_id' => $old_customer_id,
            'old_customer_name' => $old_user ? $old_user->display_name : __('Aucun', 'wc-quotes-woodmart'),
            'old_customer_email' => $old_user ? $old_user->user_email : '',
            'new_customer_id' => $new_customer_id,
            'new_customer_name' => $new_user->display_name,
            'new_customer_email' => $new_user->user_email,
            'changed_by' => $current_user->ID,
            'changed_by_name' => $current_user->display_name,
        );

        // Sauvegarder le journal
        update_post_meta($quote_id, '_customer_change_log', $change_log);
    }
    
    /**
     * Afficher un message de succès après le changement de client
     */
    public function customer_change_notice() {
        if (isset($_GET['wcq_customer_changed']) && $_GET['wcq_customer_changed'] == 1) {
            $customer_name = isset($_GET['wcq_new_customer_name']) ? urldecode($_GET['wcq_new_customer_name']) : '';
            ?>
            <div class="notice notice-success is-dismissible">
                <p>
                    <?php
                    printf(
                        __('Le devis a été attribué avec succès à <strong>%s</strong>.', 'wc-quotes-woodmart'),
                        esc_html($customer_name)
                    );
                    ?>
                </p>
            </div>
            <?php
        }
    }

    /**
     * Supprimer l'édition rapide
     */
    public function remove_quick_edit() {
        global $current_screen;

        if ($current_screen && $current_screen->post_type === 'shop_quote') {
            ?>
            <style>
                .inline-edit-row { display: none !important; }
            </style>
            <?php
        }
    }
}

// Créer une instance globale pour être accessible depuis d'autres classes
global $wcq_admin_instance;
$wcq_admin_instance = new WCQ_Admin();
