<?php
/**
 * Gestion du panier et du bouton devis
 *
 * @package WC_Quotes_Woodmart
 */

if (!defined('ABSPATH')) {
    exit;
}

class WCQ_Cart {
    
    /**
     * Constructeur
     */
    public function __construct() {
        // Ajouter le bouton "Transformer en devis" sur la page panier
        add_action('woocommerce_proceed_to_checkout', array($this, 'add_quote_button'), 5);

        // Ajouter le bouton "Transformer en devis" dans le panier flottant
        add_action('woocommerce_widget_shopping_cart_buttons', array($this, 'add_quote_button_mini_cart'), 15);

        // Gérer la création directe du devis depuis le panier
        add_action('template_redirect', array($this, 'handle_cart_to_quote'));
    }
    
    /**
     * Ajouter le bouton "Transformer en devis" sur la page panier
     */
    public function add_quote_button() {
        // Vérifier si le panier n'est pas vide
        if (WC()->cart->is_empty()) {
            return;
        }
        
        // Vérifier si l'utilisateur est connecté
        if (!is_user_logged_in()) {
            ?>
            <a href="<?php echo esc_url(wp_login_url(wc_get_cart_url())); ?>" class="button alt wcq-create-quote-button">
                <?php _e('Se connecter pour créer un devis', 'wc-quotes-woodmart'); ?>
            </a>
            <?php
            return;
        }
        
        // Générer l'URL de création de devis avec nonce
        $quote_url = wp_nonce_url(
            add_query_arg('create-quote', '1', wc_get_cart_url()),
            'create_quote_from_cart'
        );
        
        ?>
        <a href="<?php echo esc_url($quote_url); ?>" class="button alt wcq-create-quote-button">
            <?php _e('Transformer en devis', 'wc-quotes-woodmart'); ?>
        </a>
        <?php
    }

    /**
     * Ajouter le bouton "Transformer en devis" dans le panier flottant (mini cart)
     */
    public function add_quote_button_mini_cart() {
        // Vérifier si le panier n'est pas vide
        if (WC()->cart->is_empty()) {
            return;
        }

        // Vérifier si l'utilisateur est connecté
        if (!is_user_logged_in()) {
            ?>
            <a href="<?php echo esc_url(wp_login_url(wc_get_cart_url())); ?>" class="button wcq-create-quote-button-mini">
                <?php _e('Se connecter pour créer un devis', 'wc-quotes-woodmart'); ?>
            </a>
            <?php
            return;
        }

        // Générer l'URL de création de devis avec nonce
        $quote_url = wp_nonce_url(
            add_query_arg('create-quote', '1', wc_get_cart_url()),
            'create_quote_from_cart'
        );

        ?>
        <a href="<?php echo esc_url($quote_url); ?>" class="button wcq-create-quote-button-mini">
            <?php _e('Transformer en devis', 'wc-quotes-woodmart'); ?>
        </a>
        <style>
            /* Style pour le bouton dans le mini cart */
            .wcq-create-quote-button-mini {
                display: block;
                width: 100%;
                text-align: center;
                margin-top: 10px;
                background-color: #f5f5f5 !important;
                color: #333 !important;
                border: 1px solid #ddd !important;
                padding: 12px 20px !important;
                font-size: 14px !important;
                font-weight: 600 !important;
                text-transform: uppercase;
                letter-spacing: 0.5px;
                transition: all 0.3s ease;
            }

            .wcq-create-quote-button-mini:hover {
                background-color: #e0e0e0 !important;
                border-color: #bbb !important;
            }

            /* Ajuster l'espacement des boutons du mini cart */
            .woocommerce-mini-cart__buttons {
                display: flex;
                flex-direction: column;
                gap: 10px;
            }

            .woocommerce-mini-cart__buttons .button {
                margin: 0 !important;
            }
        </style>
        <?php
    }

    /**
     * Gérer la création directe du devis depuis le panier
     */
    public function handle_cart_to_quote() {
        // Vérifier si on demande la création d'un devis
        if (!isset($_GET['create-quote']) || $_GET['create-quote'] !== '1') {
            return;
        }
        
        // Vérifier le nonce
        if (!isset($_GET['_wpnonce']) || !wp_verify_nonce($_GET['_wpnonce'], 'create_quote_from_cart')) {
            wc_add_notice(__('Action non autorisée.', 'wc-quotes-woodmart'), 'error');
            wp_redirect(wc_get_cart_url());
            exit;
        }
        
        // Vérifier que l'utilisateur est connecté
        if (!is_user_logged_in()) {
            wc_add_notice(__('Vous devez être connecté pour créer un devis.', 'wc-quotes-woodmart'), 'error');
            wp_redirect(wp_login_url(wc_get_cart_url()));
            exit;
        }
        
        // Vérifier si le panier n'est pas vide
        if (WC()->cart->is_empty()) {
            wc_add_notice(__('Votre panier est vide.', 'wc-quotes-woodmart'), 'error');
            wp_redirect(wc_get_cart_url());
            exit;
        }

        // Récupérer les données du panier
        $cart_data = array();
        foreach (WC()->cart->get_cart() as $cart_item_key => $cart_item) {
            // DEBUG : Logger tout le contenu du cart_item
            error_log('=== DEBUG CART ITEM ===');
            error_log('Cart item keys: ' . implode(', ', array_keys($cart_item)));

            // Prix
            error_log('line_total: ' . (isset($cart_item['line_total']) ? $cart_item['line_total'] : 'NOT SET'));
            error_log('line_subtotal: ' . (isset($cart_item['line_subtotal']) ? $cart_item['line_subtotal'] : 'NOT SET'));

            $quote_item = array(
                'product_id'     => $cart_item['product_id'],
                'variation_id'   => isset($cart_item['variation_id']) ? $cart_item['variation_id'] : 0,
                'quantity'       => $cart_item['quantity'],
                'line_total'     => $cart_item['line_total'],
                'line_subtotal'  => $cart_item['line_subtotal'],
                'variation'      => isset($cart_item['variation']) ? $cart_item['variation'] : array(),
            );

            // CORRECTION : Si taxonomy_discounts existe, utiliser base_price comme line_subtotal
            if (isset($cart_item['taxonomy_discounts'])) {
                $tax_disc = $cart_item['taxonomy_discounts'];
                if (isset($tax_disc['base_price']) && isset($tax_disc['discount_price'])) {
                    // Le base_price est le prix AVANT remise
                    $quote_item['line_subtotal'] = floatval($tax_disc['base_price']);
                    $quote_item['line_total'] = floatval($tax_disc['discount_price']);
                    error_log("CORRECTION PRIX: line_subtotal={$quote_item['line_subtotal']}, line_total={$quote_item['line_total']}");
                }
            }

            // Méthode 1 : Capturer les métadonnées depuis $cart_item['data'] (objet produit WC)
            if (isset($cart_item['data']) && is_object($cart_item['data'])) {
                $product = $cart_item['data'];
                error_log('Product class: ' . get_class($product));
                error_log('Product ID: ' . $product->get_id());
                error_log('Parent ID: ' . ($product->is_type('variation') ? $product->get_parent_id() : 'N/A'));

                // DEBUG : Afficher TOUTES les métadonnées du produit
                $all_meta = get_post_meta($product->get_id());
                error_log('ALL PRODUCT META: ' . print_r(array_keys($all_meta), true));
                if ($product->is_type('variation')) {
                    $parent_meta = get_post_meta($product->get_parent_id());
                    error_log('ALL PARENT META: ' . print_r(array_keys($parent_meta), true));
                }

                // Essayer get_meta_data()
                if (method_exists($product, 'get_meta_data')) {
                    $meta_data = $product->get_meta_data();
                    if (!empty($meta_data)) {
                        error_log('Found ' . count($meta_data) . ' product meta_data entries');
                        foreach ($meta_data as $meta) {
                            if (is_object($meta) && method_exists($meta, 'get_data')) {
                                $data = $meta->get_data();
                                if (isset($data['key']) && isset($data['value'])) {
                                    error_log("Product meta: {$data['key']} = " . print_r($data['value'], true));
                                    $quote_item[$data['key']] = $data['value'];
                                }
                            }
                        }
                    }
                }

                // Essayer aussi get_meta() pour des clés spécifiques liées à la livraison
                $delivery_meta_keys = array(
                    '_delivery_date',
                    '_estimated_delivery',
                    'delivery_date',
                    'estimated_delivery',
                    'livraison_estimee',
                    '_livraison_estimee',
                    'shipping_date',
                    '_shipping_date',
                    'ywpd_delivery_date',
                    '_ywpd_delivery_date',
                    // Clés spécifiques au plugin "Estimated Delivery for WooCommerce"
                    '_edd_delivery_date',
                    '_edd_min_days',
                    '_edd_max_days',
                    'edd_delivery_date',
                );

                foreach ($delivery_meta_keys as $key) {
                    if (method_exists($product, 'get_meta')) {
                        $value = $product->get_meta($key, true);
                        if (!empty($value)) {
                            error_log("Found delivery meta with key '$key': " . print_r($value, true));
                            $quote_item['delivery_date'] = $value;
                            break;
                        }
                    }
                }

                // Si le plugin Estimated Delivery est actif, essayer de calculer la date
                if (empty($quote_item['delivery_date'])) {
                    // Essayer d'obtenir la date calculée par le plugin
                    $product_id_for_delivery = $product->is_type('variation') ? $product->get_parent_id() : $product->get_id();

                    // Chercher les jours min/max de livraison
                    $min_days = $product->get_meta('_edd_min_days', true);
                    $max_days = $product->get_meta('_edd_max_days', true);

                    if (empty($min_days)) {
                        $min_days = get_post_meta($product_id_for_delivery, '_edd_min_days', true);
                    }
                    if (empty($max_days)) {
                        $max_days = get_post_meta($product_id_for_delivery, '_edd_max_days', true);
                    }

                    if (!empty($max_days)) {
                        // Calculer la date de livraison estimée
                        $delivery_timestamp = strtotime("+{$max_days} days");
                        $delivery_date_formatted = date_i18n('j F Y', $delivery_timestamp);
                        $quote_item['delivery_date'] = " on " . $delivery_date_formatted;
                        error_log("Calculated delivery date from _edd_max_days ($max_days): {$quote_item['delivery_date']}");
                    }
                }
            }

            // Méthode 2 : Capturer toutes les clés personnalisées du $cart_item lui-même
            $excluded_keys = array('key', 'product_id', 'variation_id', 'variation', 'quantity',
                                   'line_total', 'line_subtotal', 'line_tax', 'line_subtotal_tax',
                                   'data', 'data_hash');

            foreach ($cart_item as $key => $value) {
                if (!in_array($key, $excluded_keys)) {
                    // Ne pas écraser les métadonnées déjà capturées
                    if (!isset($quote_item[$key])) {
                        error_log("Cart item custom key: '$key' = " . (is_scalar($value) ? $value : print_r($value, true)));
                        $quote_item[$key] = $value;
                    }
                }
            }

            error_log('Final quote_item keys: ' . implode(', ', array_keys($quote_item)));
            error_log('=== END DEBUG ===');

            $cart_data[] = $quote_item;
        }

        // DEBUG : Sauvegarder les données brutes pour inspection
        if (!empty($cart_data)) {
            error_log('CART DATA BEING SAVED: ' . print_r($cart_data, true));
        }

        // Créer le devis directement
        $quote_id = WCQ_Quote::create_quote($cart_data, get_current_user_id(), 30);

        if (!$quote_id) {
            wc_add_notice(__('Erreur lors de la création du devis. Veuillez réessayer.', 'wc-quotes-woodmart'), 'error');
            wp_redirect(wc_get_cart_url());
            exit;
        }

        // DEBUG : Afficher les clés capturées dans une notice
        if (current_user_can('manage_options') && !empty($cart_data)) {
            $first_item = $cart_data[0];
            $debug_keys = implode(', ', array_keys($first_item));
            wc_add_notice("DEBUG - Clés capturées : $debug_keys", 'notice');
        }

        // Vider le panier après création du devis (configurable via filtre)
        $empty_cart = apply_filters('wcq_empty_cart_after_quote', true);

        if ($empty_cart) {
            WC()->cart->empty_cart();
        }

        // Message de succès
        $quote = WCQ_Quote::get_quote($quote_id);
        wc_add_notice(
            sprintf(
                __('Votre devis #%s a été créé avec succès !', 'wc-quotes-woodmart'),
                $quote['quote_number']
            ),
            'success'
        );
        
        // Rediriger vers la page du devis dans l'espace client
        wp_redirect(wc_get_account_endpoint_url('quotes') . $quote_id . '/');
        exit;
    }
}

new WCQ_Cart();
