<?php
/**
 * Gestion de l'espace client
 *
 * @package WC_Quotes_Woodmart
 */

if (!defined('ABSPATH')) {
    exit;
}

class WCQ_Customer {
    
    /**
     * Constructeur
     */
    public function __construct() {
        // Ajouter l'endpoint "Mes devis" au compte client
        add_action('init', array($this, 'add_quotes_endpoint'));

        // Ajouter l'onglet dans le menu du compte
        add_filter('woocommerce_account_menu_items', array($this, 'add_quotes_menu_item'), 40);

        // Afficher le contenu de la page "Mes devis"
        add_action('woocommerce_account_quotes_endpoint', array($this, 'quotes_content'));

        // Gérer les actions sur les devis (convert, delete, load_to_cart, etc.)
        add_action('template_redirect', array($this, 'handle_quote_actions'));

        // Gérer la conversion de devis en commande depuis l'espace client
        add_action('template_redirect', array($this, 'handle_customer_convert_quote'));

        // Gérer la mise à jour des devis (quantités et suppressions)
        add_action('template_redirect', array($this, 'handle_update_quote'));

        // Gérer le transfert de devis par les admins depuis le frontend
        add_action('template_redirect', array($this, 'handle_admin_transfer_quote'));

        // Forcer le flush des rewrite rules si nécessaire
        add_action('init', array($this, 'check_rewrite_rules'), 999);

        // Ajouter le script pour afficher la pastille de notification
        add_action('wp_footer', array($this, 'add_unread_badge_script'));
    }
    
    /**
     * Vérifier et régénérer les règles de réécriture si nécessaire
     */
    public function check_rewrite_rules() {
        // Vérifier si on a déjà flush les règles pour cette version
        $flushed = get_option('wcq_customer_rewrite_flushed', '0');
        if ($flushed !== WCQ_VERSION) {
            flush_rewrite_rules();
            update_option('wcq_customer_rewrite_flushed', WCQ_VERSION);
        }
    }
    
    /**
     * Ajouter l'endpoint "quotes" au compte client
     */
    public function add_quotes_endpoint() {
        // Méthode 1 : Via les options WooCommerce (recommandé)
        add_option('woocommerce_myaccount_quotes_endpoint', 'quotes');
        
        // Méthode 2 : Via add_rewrite_endpoint (backup)
        add_rewrite_endpoint('quotes', EP_ROOT | EP_PAGES);
    }
    
    /**
     * Ajouter l'élément de menu "Mes devis"
     */
    public function add_quotes_menu_item($items) {
        // Insérer avant "Déconnexion"
        $logout = $items['customer-logout'];
        unset($items['customer-logout']);

        $items['quotes'] = __('Mes devis', 'wc-quotes-woodmart');
        $items['customer-logout'] = $logout;

        return $items;
    }
    
    /**
     * Afficher le contenu de la page "Mes devis"
     */
    public function quotes_content() {
        // Vérifier si on affiche un devis spécifique
        global $wp;
        
        if (!empty($wp->query_vars['quotes'])) {
            $quote_id = absint($wp->query_vars['quotes']);
            if ($quote_id > 0) {
                $this->single_quote_content($quote_id);
                return;
            }
        }
        
        // Afficher la liste des devis
        $this->quotes_list_content();
    }
    
    /**
     * Afficher la liste des devis
     */
    private function quotes_list_content() {
        $user_id = get_current_user_id();
        $quotes = WCQ_Quote::get_user_quotes($user_id);
        
        wc_get_template(
            'myaccount/quotes-list.php',
            array('quotes' => $quotes),
            '',
            WCQ_PLUGIN_DIR . 'templates/'
        );
    }
    
    /**
     * Afficher un devis unique
     */
    private function single_quote_content($quote_id) {
        $quote = WCQ_Quote::get_quote($quote_id);

        // Vérifier que le devis appartient à l'utilisateur
        if (!$quote || $quote['customer_id'] != get_current_user_id()) {
            wc_add_notice(__('Ce devis n\'existe pas ou ne vous appartient pas.', 'wc-quotes-woodmart'), 'error');
            wp_redirect(wc_get_account_endpoint_url('quotes'));
            exit;
        }

        // Marquer le devis comme lu
        $this->mark_quote_as_read($quote_id);

        wc_get_template(
            'myaccount/quote-single.php',
            array('quote' => $quote),
            '',
            WCQ_PLUGIN_DIR . 'templates/'
        );
    }
    
    /**
     * Gérer la conversion de devis en commande depuis l'espace client
     */
    public function handle_customer_convert_quote() {
        if (!isset($_GET['convert-quote']) || !isset($_GET['quote_id'])) {
            return;
        }
        
        // Vérifier le nonce
        if (!isset($_GET['_wpnonce']) || !wp_verify_nonce($_GET['_wpnonce'], 'convert_quote')) {
            wc_add_notice(__('Action non autorisée', 'wc-quotes-woodmart'), 'error');
            wp_redirect(wc_get_account_endpoint_url('quotes'));
            exit;
        }
        
        $quote_id = absint($_GET['quote_id']);
        $quote = WCQ_Quote::get_quote($quote_id);
        
        // Vérifier que le devis appartient à l'utilisateur
        if (!$quote || $quote['customer_id'] != get_current_user_id()) {
            wc_add_notice(__('Ce devis n\'existe pas ou ne vous appartient pas.', 'wc-quotes-woodmart'), 'error');
            wp_redirect(wc_get_account_endpoint_url('quotes'));
            exit;
        }
        
        // Vérifier que le devis n'est pas expiré
        if ($quote['valid_until']) {
            $valid_date = strtotime($quote['valid_until']);
            if ($valid_date < current_time('timestamp')) {
                wc_add_notice(__('Ce devis a expiré et ne peut plus être converti en commande.', 'wc-quotes-woodmart'), 'error');
                wp_redirect(wc_get_account_endpoint_url('quotes'));
                exit;
            }
        }
        
        // Vider le panier actuel
        WC()->cart->empty_cart();
        
        // Ajouter les produits du devis au panier
        foreach ($quote['quote_data'] as $cart_item) {
            $product_id = $cart_item['product_id'];
            $variation_id = isset($cart_item['variation_id']) ? $cart_item['variation_id'] : 0;
            $quantity = $cart_item['quantity'];
            $variation = isset($cart_item['variation']) ? $cart_item['variation'] : array();
            
            WC()->cart->add_to_cart($product_id, $quantity, $variation_id, $variation);
        }
        
        // Ajouter une note
        wc_add_notice(
            sprintf(__('Les produits du devis #%s ont été ajoutés à votre panier.', 'wc-quotes-woodmart'), $quote['quote_number']),
            'success'
        );
        
        // Rediriger vers le panier
        wp_redirect(wc_get_checkout_url());
        exit;
    }
    
    /**
    /**
     * Gérer les actions sur les devis (GET parameters)
     */
    public function handle_quote_actions() {
        if (!isset($_GET['wcq_action']) || !isset($_GET['quote_id'])) {
            return;
        }
        
        if (!is_user_logged_in()) {
            return;
        }
        
        $action = sanitize_text_field($_GET['wcq_action']);
        $quote_id = intval($_GET['quote_id']);
        
        // Récupérer le devis
        $quote = get_post($quote_id);
        
        if (!$quote || $quote->post_type !== 'shop_quote') {
            wc_add_notice(__('Devis introuvable.', 'wc-quotes-woodmart'), 'error');
            return;
        }
        
        // Vérifier que c'est le bon client
        $customer_id = get_post_meta($quote_id, '_customer_id', true);
        if ($customer_id != get_current_user_id()) {
            wc_add_notice(__('Vous n\'avez pas accès à ce devis.', 'wc-quotes-woodmart'), 'error');
            return;
        }
        
        switch ($action) {
            case 'convert':
                // Vérifier le nonce
                if (!isset($_GET['_wpnonce']) || !wp_verify_nonce($_GET['_wpnonce'], 'wcq_convert_' . $quote_id)) {
                    wc_add_notice(__('Action non autorisée.', 'wc-quotes-woodmart'), 'error');
                    return;
                }
                
                // Convertir le devis en commande
                $this->convert_quote_to_cart($quote_id, true);
                break;
                
            case 'load_to_cart':
                // Vérifier le nonce
                if (!isset($_GET['_wpnonce']) || !wp_verify_nonce($_GET['_wpnonce'], 'wcq_load_' . $quote_id)) {
                    wc_add_notice(__('Action non autorisée.', 'wc-quotes-woodmart'), 'error');
                    return;
                }
                
                // Rediriger vers la page d'édition de devis
                $edit_page_id = get_option('wcq_edit_quote_page_id');
                if ($edit_page_id) {
                    wp_redirect(add_query_arg('quote_id', $quote_id, get_permalink($edit_page_id)));
                    exit;
                } else {
                    // Fallback : charger dans le panier normal
                    $this->load_quote_to_cart($quote_id);
                }
                break;
                
            case 'update_from_cart':
                // Vérifier le nonce
                if (!isset($_GET['_wpnonce']) || !wp_verify_nonce($_GET['_wpnonce'], 'wcq_update_' . $quote_id)) {
                    wc_add_notice(__('Action non autorisée.', 'wc-quotes-woodmart'), 'error');
                    return;
                }
                
                // Mettre à jour le devis avec le contenu du panier
                $this->update_quote_from_cart($quote_id);
                break;
                
            case 'delete':
                // Vérifier le nonce
                if (!isset($_GET['_wpnonce']) || !wp_verify_nonce($_GET['_wpnonce'], 'wcq_delete_' . $quote_id)) {
                    wc_add_notice(__('Action non autorisée.', 'wc-quotes-woodmart'), 'error');
                    return;
                }
                
                // Supprimer le devis
                wp_trash_post($quote_id);
                wc_add_notice(__('Le devis a été supprimé.', 'wc-quotes-woodmart'), 'success');
                wp_redirect(wc_get_account_endpoint_url('quotes'));
                exit;
                break;
                
            case 'download_pdf':
                // Télécharger le PDF
                if (class_exists('WCQ_PDF')) {
                    WCQ_PDF::generate_pdf($quote_id, true);
                }
                exit;
                break;
                
            case 'send_email':
                // Envoyer par email
                if (class_exists('WCQ_Email')) {
                    $email = new WCQ_Email();
                    $sent = $email->send_quote_to_customer($quote_id);
                    if ($sent) {
                        wc_add_notice(__('Le devis vous a été envoyé par email.', 'wc-quotes-woodmart'), 'success');
                    } else {
                        wc_add_notice(__('Erreur lors de l\'envoi de l\'email.', 'wc-quotes-woodmart'), 'error');
                    }
                }
                wp_redirect(wc_get_account_endpoint_url('quotes') . $quote_id . '/');
                exit;
                break;
        }
    }
    
    /**
     * Convertir le devis en panier
     */
    private function convert_quote_to_cart($quote_id, $redirect_to_checkout = false) {
        // Vider le panier actuel
        WC()->cart->empty_cart();
        
        // Récupérer les articles du devis
        $quote_data = get_post_meta($quote_id, '_quote_data', true);
        
        if (empty($quote_data)) {
            wc_add_notice(__('Le devis est vide.', 'wc-quotes-woodmart'), 'error');
            return;
        }
        
        // Ajouter chaque article au panier
        foreach ($quote_data as $item) {
            $product_id = $item['product_id'];
            $quantity = $item['quantity'];
            $variation_id = isset($item['variation_id']) ? $item['variation_id'] : 0;
            $variation = isset($item['variation']) ? $item['variation'] : array();
            
            WC()->cart->add_to_cart($product_id, $quantity, $variation_id, $variation);
        }
        
        wc_add_notice(__('Les articles du devis ont été ajoutés au panier.', 'wc-quotes-woodmart'), 'success');
        
        // Rediriger
        if ($redirect_to_checkout) {
            wp_redirect(wc_get_checkout_url());
        } else {
            wp_redirect(wc_get_cart_url());
        }
        exit;
    }
    
    /**
     * Charger le devis dans le panier pour modification
     */
    private function load_quote_to_cart($quote_id) {
        // Vider le panier actuel
        WC()->cart->empty_cart();
        
        // Récupérer les articles du devis
        $quote_data = get_post_meta($quote_id, '_quote_data', true);
        
        if (empty($quote_data)) {
            wc_add_notice(__('Le devis est vide.', 'wc-quotes-woodmart'), 'error');
            wp_redirect(wc_get_account_endpoint_url('quotes') . $quote_id . '/');
            exit;
        }
        
        // Ajouter chaque article au panier
        foreach ($quote_data as $item) {
            $product_id = $item['product_id'];
            $quantity = $item['quantity'];
            $variation_id = isset($item['variation_id']) ? $item['variation_id'] : 0;
            $variation = isset($item['variation']) ? $item['variation'] : array();
            
            WC()->cart->add_to_cart($product_id, $quantity, $variation_id, $variation);
        }
        
        // Message avec instructions
        wc_add_notice(__('Le devis a été chargé dans votre panier. Modifiez les quantités puis créez un nouveau devis.', 'wc-quotes-woodmart'), 'success');
        
        // Rediriger vers le panier
        wp_redirect(wc_get_cart_url());
        exit;
    }
    
    /**
     * Mettre à jour le devis avec le contenu du panier
     */
    private function update_quote_from_cart($quote_id) {
        // Vérifier que le panier n'est pas vide
        if (WC()->cart->is_empty()) {
            wc_add_notice(__('Le panier est vide.', 'wc-quotes-woodmart'), 'error');
            wp_redirect(wc_get_account_endpoint_url('quotes') . $quote_id . '/');
            exit;
        }
        
        // Récupérer l'ancien devis
        $old_quote = get_post($quote_id);
        $old_quote_number = get_post_meta($quote_id, '_quote_number', true);
        
        // Préparer les nouvelles données depuis le panier
        $new_quote_data = array();
        $new_total = 0;
        
        foreach (WC()->cart->get_cart() as $cart_item) {
            $product_id = $cart_item['product_id'];
            $variation_id = $cart_item['variation_id'];
            $quantity = $cart_item['quantity'];
            $line_total = $cart_item['line_total'];
            
            $new_quote_data[] = array(
                'product_id' => $product_id,
                'variation_id' => $variation_id,
                'variation' => isset($cart_item['variation']) ? $cart_item['variation'] : array(),
                'quantity' => $quantity,
                'line_total' => $line_total,
            );
            
            $new_total += $line_total;
        }
        
        // Calculer le numéro de version
        $version_number = $this->get_next_version_number($old_quote_number);
        
        // Créer le nouveau devis
        $new_quote_id = wp_insert_post(array(
            'post_type'   => 'shop_quote',
            'post_status' => 'quote-pending',
            'post_title'  => sprintf(__('Devis #%s', 'wc-quotes-woodmart'), $version_number),
            'post_author' => get_current_user_id(),
        ));
        
        if (is_wp_error($new_quote_id)) {
            wc_add_notice(__('Erreur lors de la création du devis.', 'wc-quotes-woodmart'), 'error');
            return;
        }
        
        // Copier les métadonnées
        $valid_until = get_post_meta($quote_id, '_valid_until', true);

        update_post_meta($new_quote_id, '_quote_number', $version_number);
        update_post_meta($new_quote_id, '_quote_data', $new_quote_data);
        update_post_meta($new_quote_id, '_quote_total', $new_total);
        update_post_meta($new_quote_id, '_valid_until', $valid_until);
        update_post_meta($new_quote_id, '_customer_id', get_current_user_id());
        update_post_meta($new_quote_id, '_parent_quote_id', $quote_id);
        update_post_meta($new_quote_id, '_original_quote_number', $old_quote_number);
        update_post_meta($new_quote_id, '_is_read', '1'); // Marqué comme lu car créé par le client
        
        // Copier les infos client
        $customer_email = get_post_meta($quote_id, '_customer_email', true);
        $customer_first_name = get_post_meta($quote_id, '_customer_first_name', true);
        $customer_last_name = get_post_meta($quote_id, '_customer_last_name', true);
        
        update_post_meta($new_quote_id, '_customer_email', $customer_email);
        update_post_meta($new_quote_id, '_customer_first_name', $customer_first_name);
        update_post_meta($new_quote_id, '_customer_last_name', $customer_last_name);
        
        // Marquer l'ancien devis comme "remplacé"
        wp_update_post(array(
            'ID' => $quote_id,
            'post_status' => 'quote-replaced'
        ));
        update_post_meta($quote_id, '_replaced_by_quote_id', $new_quote_id);
        
        // Vider le panier
        WC()->cart->empty_cart();
        
        // Nettoyer la session
        WC()->session->set('editing_quote_id', null);
        
        // Message de succès
        wc_add_notice(sprintf(
            __('Le devis a été mis à jour. Nouveau numéro : %s', 'wc-quotes-woodmart'),
            $version_number
        ), 'success');
        
        // Rediriger vers le nouveau devis
        wp_redirect(wc_get_account_endpoint_url('quotes') . $new_quote_id . '/');
        exit;
    }
    
    /**
     * Gérer la mise à jour du devis (en créant une nouvelle version)
     */
    public function handle_update_quote() {
        // Vérifier si c'est une demande de mise à jour
        if (!isset($_POST['wcq_action']) || $_POST['wcq_action'] !== 'update_quote') {
            return;
        }
        
        // Vérifier le nonce
        $quote_id = isset($_POST['quote_id']) ? intval($_POST['quote_id']) : 0;
        if (!isset($_POST['wcq_update_nonce']) || !wp_verify_nonce($_POST['wcq_update_nonce'], 'wcq_update_quote_' . $quote_id)) {
            wc_add_notice(__('Action non autorisée.', 'wc-quotes-woodmart'), 'error');
            return;
        }
        
        // Vérifier que l'utilisateur est connecté
        if (!is_user_logged_in()) {
            wc_add_notice(__('Vous devez être connecté.', 'wc-quotes-woodmart'), 'error');
            return;
        }
        
        // Récupérer le devis original
        $original_quote = get_post($quote_id);
        
        if (!$original_quote || $original_quote->post_type !== 'shop_quote') {
            wc_add_notice(__('Devis introuvable.', 'wc-quotes-woodmart'), 'error');
            return;
        }
        
        // Vérifier que c'est le bon client
        $customer_id = get_post_meta($quote_id, '_customer_id', true);
        if ($customer_id != get_current_user_id()) {
            wc_add_notice(__('Vous n\'avez pas accès à ce devis.', 'wc-quotes-woodmart'), 'error');
            return;
        }
        
        // Vérifier que le devis est en attente
        if ($original_quote->post_status !== 'quote-pending') {
            wc_add_notice(__('Seuls les devis en attente peuvent être modifiés.', 'wc-quotes-woodmart'), 'error');
            return;
        }
        
        // Récupérer les articles
        $items = isset($_POST['items']) ? $_POST['items'] : array();
        
        if (empty($items)) {
            wc_add_notice(__('Aucun article trouvé.', 'wc-quotes-woodmart'), 'error');
            return;
        }
        
        // Récupérer les données originales
        $original_quote_data = get_post_meta($quote_id, '_quote_data', true);
        $original_quote_number = get_post_meta($quote_id, '_quote_number', true);
        
        // Traiter les modifications
        $new_quote_data = array();
        $new_total = 0;
        $has_changes = false;
        
        foreach ($items as $index => $item) {
            // Vérifier si l'article doit être supprimé
            if (isset($item['remove']) && $item['remove'] == '1') {
                $has_changes = true;
                continue; // Ne pas ajouter cet article
            }
            
            // Vérifier que l'index existe dans les données originales
            if (!isset($original_quote_data[$index])) {
                continue;
            }
            
            $original_item = $original_quote_data[$index];
            $new_quantity = isset($item['quantity']) ? intval($item['quantity']) : $original_item['quantity'];
            
            // Vérifier que la quantité est valide
            if ($new_quantity < 1) {
                $new_quantity = 1;
            }
            
            // Vérifier si la quantité a changé
            if ($new_quantity != $original_item['quantity']) {
                $has_changes = true;
            }
            
            // Calculer le nouveau total de ligne
            $unit_price = $original_item['line_total'] / $original_item['quantity'];
            $line_total = $unit_price * $new_quantity;
            
            // Créer le nouvel article
            $new_item = $original_item;
            $new_item['quantity'] = $new_quantity;
            $new_item['line_total'] = $line_total;
            
            $new_quote_data[] = $new_item;
            $new_total += $line_total;
        }
        
        // Vérifier qu'il reste au moins un article
        if (empty($new_quote_data)) {
            wc_add_notice(__('Le devis doit contenir au moins un article.', 'wc-quotes-woodmart'), 'error');
            wp_redirect(wc_get_account_endpoint_url('quotes') . $quote_id . '/');
            exit;
        }
        
        // Si pas de changements, ne rien faire
        if (!$has_changes) {
            wc_add_notice(__('Aucune modification détectée.', 'wc-quotes-woodmart'), 'notice');
            wp_redirect(wc_get_account_endpoint_url('quotes') . $quote_id . '/');
            exit;
        }
        
        // Calculer le numéro de version
        $version_number = $this->get_next_version_number($original_quote_number);
        
        // Créer le nouveau devis (version modifiée)
        $new_quote_id = wp_insert_post(array(
            'post_type'   => 'shop_quote',
            'post_status' => 'quote-pending',
            'post_title'  => sprintf(__('Devis #%s', 'wc-quotes-woodmart'), $version_number),
            'post_author' => get_current_user_id(),
        ));
        
        if (is_wp_error($new_quote_id)) {
            wc_add_notice(__('Erreur lors de la création du devis.', 'wc-quotes-woodmart'), 'error');
            return;
        }
        
        // Copier les métadonnées
        $valid_until = get_post_meta($quote_id, '_valid_until', true);

        update_post_meta($new_quote_id, '_quote_number', $version_number);
        update_post_meta($new_quote_id, '_quote_data', $new_quote_data);
        update_post_meta($new_quote_id, '_quote_total', $new_total);
        update_post_meta($new_quote_id, '_valid_until', $valid_until);
        update_post_meta($new_quote_id, '_customer_id', get_current_user_id());
        update_post_meta($new_quote_id, '_parent_quote_id', $quote_id); // Lien vers l'original
        update_post_meta($new_quote_id, '_original_quote_number', $original_quote_number);
        update_post_meta($new_quote_id, '_is_read', '1'); // Marqué comme lu car créé par le client
        
        // Copier les infos client
        $customer_email = get_post_meta($quote_id, '_customer_email', true);
        $customer_first_name = get_post_meta($quote_id, '_customer_first_name', true);
        $customer_last_name = get_post_meta($quote_id, '_customer_last_name', true);
        
        update_post_meta($new_quote_id, '_customer_email', $customer_email);
        update_post_meta($new_quote_id, '_customer_first_name', $customer_first_name);
        update_post_meta($new_quote_id, '_customer_last_name', $customer_last_name);
        
        // Marquer l'ancien devis comme "remplacé"
        wp_update_post(array(
            'ID' => $quote_id,
            'post_status' => 'quote-replaced'
        ));
        update_post_meta($quote_id, '_replaced_by_quote_id', $new_quote_id);
        
        // Message de succès
        wc_add_notice(sprintf(
            __('Le devis a été mis à jour. Nouveau numéro : %s', 'wc-quotes-woodmart'),
            $version_number
        ), 'success');
        
        // Rediriger vers le nouveau devis
        wp_redirect(wc_get_account_endpoint_url('quotes') . $new_quote_id . '/');
        exit;
    }
    
    /**
     * Calculer le prochain numéro de version
     */
    private function get_next_version_number($original_number) {
        // Extraire le numéro de base (DEV-000001)
        $base_number = $original_number;

        // Chercher si c'est déjà une version (DEV-000001-MAJ001)
        if (preg_match('/^(.+)-MAJ(\d+)$/', $original_number, $matches)) {
            $base_number = $matches[1];
            $version = intval($matches[2]);
        } else {
            $version = 0;
        }

        // Incrémenter la version
        $version++;

        // Retourner le nouveau numéro avec version
        return $base_number . '-MAJ' . str_pad($version, 3, '0', STR_PAD_LEFT);
    }

    /**
     * Compter le nombre de devis non lus pour l'utilisateur courant
     */
    private function get_unread_quotes_count() {
        $user_id = get_current_user_id();

        if (!$user_id) {
            return 0;
        }

        // Récupérer tous les devis en attente de l'utilisateur
        $args = array(
            'post_type'      => 'shop_quote',
            'post_status'    => 'quote-pending',
            'posts_per_page' => -1,
            'author'         => $user_id,
            'meta_query'     => array(
                'relation' => 'OR',
                array(
                    'key'     => '_is_read',
                    'compare' => 'NOT EXISTS'
                ),
                array(
                    'key'     => '_is_read',
                    'value'   => '1',
                    'compare' => '!='
                )
            )
        );

        $quotes = get_posts($args);
        return count($quotes);
    }

    /**
     * Marquer un devis comme lu
     */
    private function mark_quote_as_read($quote_id) {
        update_post_meta($quote_id, '_is_read', '1');
    }

    /**
     * Ajouter le script JavaScript pour afficher la pastille de notification
     */
    public function add_unread_badge_script() {
        // Vérifier qu'on est bien sur la page Mon Compte
        if (!is_account_page()) {
            return;
        }

        $unread_count = $this->get_unread_quotes_count();

        // Ne rien afficher si pas de devis non lus
        if ($unread_count <= 0) {
            return;
        }

        ?>
        <script type="text/javascript">
        (function() {
            var unreadCount = <?php echo intval($unread_count); ?>;
            if (unreadCount > 0) {
                var badge = '<span class="wcq-unread-badge">' + unreadCount + '</span>';

                // Ajouter la pastille au menu WooCommerce standard
                var wcMenuItem = document.querySelector('.woocommerce-MyAccount-navigation-link--quotes a');
                if (wcMenuItem) {
                    wcMenuItem.innerHTML += ' ' + badge;
                }

                // Ajouter la pastille au menu Woodmart
                var woodmartMenuItem = document.querySelector('.wd-my-acc-quotes .nav-link-text');
                if (woodmartMenuItem) {
                    woodmartMenuItem.innerHTML += ' ' + badge;
                }
            }
        })();
        </script>
        <?php
    }

    /**
     * Gérer le transfert de devis par un administrateur depuis le frontend
     */
    public function handle_admin_transfer_quote() {
        // Vérifier que c'est bien une demande de transfert
        if (!isset($_POST['wcq_action']) || $_POST['wcq_action'] !== 'admin_transfer') {
            return;
        }

        // Vérifier les permissions
        if (!current_user_can('manage_woocommerce')) {
            wc_add_notice(__('Vous n\'avez pas la permission de transférer des devis.', 'wc-quotes-woodmart'), 'error');
            wp_redirect(wc_get_account_endpoint_url('quotes'));
            exit;
        }

        // Vérifier le nonce
        if (!isset($_POST['wcq_transfer_nonce']) || !wp_verify_nonce($_POST['wcq_transfer_nonce'], 'wcq_admin_transfer_quote')) {
            wc_add_notice(__('Action non autorisée', 'wc-quotes-woodmart'), 'error');
            wp_redirect(wc_get_account_endpoint_url('quotes'));
            exit;
        }

        $quote_id = isset($_POST['quote_id']) ? intval($_POST['quote_id']) : 0;
        $new_customer_id = isset($_POST['new_customer_id']) ? intval($_POST['new_customer_id']) : 0;

        if (!$quote_id || !$new_customer_id) {
            wc_add_notice(__('Données invalides', 'wc-quotes-woodmart'), 'error');
            wp_redirect(wc_get_account_endpoint_url('quotes'));
            exit;
        }

        // Récupérer le devis
        $quote = WCQ_Quote::get_quote($quote_id);
        if (!$quote) {
            wc_add_notice(__('Devis introuvable', 'wc-quotes-woodmart'), 'error');
            wp_redirect(wc_get_account_endpoint_url('quotes'));
            exit;
        }

        $old_customer_id = $quote['customer_id'];

        // Vérifier que le nouveau client est différent
        if ($new_customer_id === intval($old_customer_id)) {
            wc_add_notice(__('Le devis appartient déjà à ce client', 'wc-quotes-woodmart'), 'error');
            wp_redirect(wc_get_account_endpoint_url('quotes') . $quote_id . '/');
            exit;
        }

        // Vérifier que le nouveau client existe
        $new_user = get_userdata($new_customer_id);
        if (!$new_user) {
            wc_add_notice(__('Client introuvable', 'wc-quotes-woodmart'), 'error');
            wp_redirect(wc_get_account_endpoint_url('quotes') . $quote_id . '/');
            exit;
        }

        // Appeler la logique de transfert (utiliser la classe WCQ_Admin)
        // Créer une instance de WCQ_Admin pour accéder à la méthode publique
        // Note: WCQ_Admin est déjà instancié via new WCQ_Admin() à la fin du fichier class-wcq-admin.php
        // On peut donc récupérer l'instance ou en créer une temporaire
        global $wcq_admin_instance;
        if (!isset($wcq_admin_instance)) {
            // Charger la classe si nécessaire
            if (!class_exists('WCQ_Admin')) {
                require_once WCQ_PLUGIN_DIR . 'includes/class-wcq-admin.php';
            }
            $wcq_admin_instance = new WCQ_Admin();
        }

        $new_quote_id = $wcq_admin_instance->transfer_quote($quote_id, $new_customer_id);

        if ($new_quote_id) {
            wc_add_notice(sprintf(
                __('Le devis a été transféré avec succès à %s. Nouveau devis créé : #%s', 'wc-quotes-woodmart'),
                $new_user->display_name,
                get_post_meta($new_quote_id, '_quote_number', true)
            ), 'success');
        } else {
            wc_add_notice(__('Erreur lors du transfert du devis', 'wc-quotes-woodmart'), 'error');
        }

        // Redirection vers le devis original
        wp_redirect(wc_get_account_endpoint_url('quotes') . $quote_id . '/');
        exit;
    }
}

new WCQ_Customer();
