<?php
/**
 * Gestion des e-mails de devis
 *
 * @package WC_Quotes_Woodmart
 */

if (!defined('ABSPATH')) {
    exit;
}

class WCQ_Email {
    
    /**
     * Constructeur
     */
    public function __construct() {
        // AJAX pour envoyer un devis par e-mail
        add_action('wp_ajax_wcq_send_quote_email', array($this, 'ajax_send_quote_email'));
        add_action('wp_ajax_nopriv_wcq_send_quote_email', array($this, 'ajax_send_quote_email'));
    }
    
    /**
     * AJAX : Envoyer un devis par e-mail
     */
    public function ajax_send_quote_email() {
        check_ajax_referer('wcq_nonce', 'nonce');
        
        $quote_id = isset($_POST['quote_id']) ? absint($_POST['quote_id']) : 0;
        $recipient_email = isset($_POST['email']) ? sanitize_email($_POST['email']) : '';
        
        if (!$quote_id) {
            wp_send_json_error(array(
                'message' => __('ID de devis invalide', 'wc-quotes-woodmart')
            ));
        }
        
        $quote = WCQ_Quote::get_quote($quote_id);
        
        if (!$quote) {
            wp_send_json_error(array(
                'message' => __('Devis introuvable', 'wc-quotes-woodmart')
            ));
        }
        
        // Vérifier les permissions
        if (!current_user_can('manage_woocommerce') && $quote['customer_id'] != get_current_user_id()) {
            wp_send_json_error(array(
                'message' => __('Permissions insuffisantes', 'wc-quotes-woodmart')
            ));
        }
        
        // Si aucun e-mail n'est spécifié, utiliser l'e-mail du client
        if (empty($recipient_email)) {
            if ($quote['customer_id'] > 0) {
                $user = get_userdata($quote['customer_id']);
                $recipient_email = $user->user_email;
            } else {
                wp_send_json_error(array(
                    'message' => __('Adresse e-mail invalide', 'wc-quotes-woodmart')
                ));
            }
        }
        
        // Envoyer l'e-mail
        $sent = $this->send_quote_email($quote_id, $recipient_email);
        
        if ($sent) {
            wp_send_json_success(array(
                'message' => sprintf(__('Devis envoyé avec succès à %s', 'wc-quotes-woodmart'), $recipient_email)
            ));
        } else {
            wp_send_json_error(array(
                'message' => __('Erreur lors de l\'envoi de l\'e-mail', 'wc-quotes-woodmart')
            ));
        }
    }
    
    /**
     * Envoyer un devis par e-mail
     */
    public function send_quote_email($quote_id, $recipient_email) {
        $quote = WCQ_Quote::get_quote($quote_id);
        
        if (!$quote) {
            return false;
        }
        
        // Récupérer les informations du site
        $site_name = get_bloginfo('name');
        
        // Récupérer les informations du client
        $customer_name = '';
        if ($quote['customer_id'] > 0) {
            $user = get_userdata($quote['customer_id']);
            if ($user) {
                $customer_name = $user->display_name;
            }
        }
        
        // Préparer le sujet de l'e-mail
        $subject = sprintf(
            __('Votre devis #%s de %s', 'wc-quotes-woodmart'),
            $quote['quote_number'],
            $site_name
        );
        
        // Préparer le contenu de l'e-mail
        $message = $this->get_email_template($quote, $customer_name);
        
        // Headers
        $headers = array(
            'Content-Type: text/html; charset=UTF-8',
            'From: ' . $site_name . ' <' . get_option('admin_email') . '>'
        );
        
        // Envoyer l'e-mail
        return wp_mail($recipient_email, $subject, $message, $headers);
    }
    
    /**
     * Obtenir le template d'e-mail
     */
    private function get_email_template($quote, $customer_name) {
        $site_name = get_bloginfo('name');
        $site_url = get_bloginfo('url');
        
        ob_start();
        ?>
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset="UTF-8">
            <meta name="viewport" content="width=device-width, initial-scale=1.0">
            <style>
                body {
                    font-family: Arial, sans-serif;
                    line-height: 1.6;
                    color: #333;
                    background-color: #f4f4f4;
                    margin: 0;
                    padding: 0;
                }
                .container {
                    max-width: 600px;
                    margin: 20px auto;
                    background: #fff;
                    padding: 20px;
                    border-radius: 5px;
                    box-shadow: 0 0 10px rgba(0,0,0,0.1);
                }
                .header {
                    background: #333;
                    color: #fff;
                    padding: 20px;
                    text-align: center;
                    border-radius: 5px 5px 0 0;
                    margin: -20px -20px 20px -20px;
                }
                .header h1 {
                    margin: 0;
                    font-size: 24px;
                }
                .quote-info {
                    background: #f9f9f9;
                    padding: 15px;
                    border-left: 4px solid #333;
                    margin: 20px 0;
                }
                table {
                    width: 100%;
                    border-collapse: collapse;
                    margin: 20px 0;
                }
                table th {
                    background: #333;
                    color: #fff;
                    padding: 10px;
                    text-align: left;
                }
                table td {
                    padding: 10px;
                    border-bottom: 1px solid #ddd;
                }
                .total {
                    font-size: 18px;
                    font-weight: bold;
                    text-align: right;
                    padding: 15px 0;
                    border-top: 2px solid #333;
                    margin-top: 10px;
                }
                .button {
                    display: inline-block;
                    padding: 12px 30px;
                    background: #333;
                    color: #fff;
                    text-decoration: none;
                    border-radius: 3px;
                    margin: 20px 0;
                }
                .footer {
                    text-align: center;
                    margin-top: 30px;
                    padding-top: 20px;
                    border-top: 1px solid #ddd;
                    font-size: 12px;
                    color: #666;
                }
            </style>
        </head>
        <body>
            <div class="container">
                <div class="header">
                    <h1><?php echo esc_html($site_name); ?></h1>
                </div>
                
                <p>Bonjour <?php echo esc_html($customer_name); ?>,</p>
                
                <p>Nous vous remercions de votre intérêt pour nos produits. Vous trouverez ci-dessous le détail de votre devis.</p>
                
                <div class="quote-info">
                    <strong>Devis #<?php echo esc_html($quote['quote_number']); ?></strong><br>
                    Date d'émission : <?php echo date_i18n(get_option('date_format'), strtotime($quote['created_at'])); ?><br>
                    <?php if ($quote['valid_until']) : ?>
                        Valide jusqu'au : <?php echo date_i18n(get_option('date_format'), strtotime($quote['valid_until'])); ?>
                    <?php endif; ?>
                </div>
                
                <table>
                    <thead>
                        <tr>
                            <th>Produit</th>
                            <th style="text-align: right;">Quantité</th>
                            <th style="text-align: right;">Prix</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php 
                        foreach ($quote['quote_data'] as $item) :
                            $product_id = isset($item['variation_id']) && $item['variation_id'] > 0 ? $item['variation_id'] : $item['product_id'];
                            $product = wc_get_product($product_id);
                            
                            if (!$product) continue;
                        ?>
                        <tr>
                            <td>
                                <?php echo esc_html($product->get_name()); ?>
                                <?php if (isset($item['variation']) && !empty($item['variation'])) : ?>
                                    <br><small style="color: #666;">
                                    <?php 
                                    $variations = array();
                                    foreach ($item['variation'] as $key => $value) {
                                        $variations[] = wc_attribute_label($key) . ': ' . $value;
                                    }
                                    echo esc_html(implode(', ', $variations));
                                    ?>
                                    </small>
                                <?php endif; ?>
                            </td>
                            <td style="text-align: right;"><?php echo esc_html($item['quantity']); ?></td>
                            <td style="text-align: right;"><?php echo wc_price($item['line_total']); ?></td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
                
                <div class="total">
                    Total : <?php echo wc_price($quote['total']); ?>
                </div>
                
                <?php if (is_user_logged_in()) : ?>
                <div style="text-align: center;">
                    <a href="<?php echo esc_url(wc_get_account_endpoint_url('quotes') . $quote['id'] . '/'); ?>" class="button">
                        Voir mon devis
                    </a>
                </div>
                <?php endif; ?>
                
                <p>Si vous avez des questions concernant ce devis, n'hésitez pas à nous contacter.</p>
                
                <div class="footer">
                    <p><?php echo esc_html($site_name); ?><br>
                    <a href="<?php echo esc_url($site_url); ?>" style="color: #666;"><?php echo esc_url($site_url); ?></a></p>
                </div>
            </div>
        </body>
        </html>
        <?php
        
        return ob_get_clean();
    }
}

new WCQ_Email();
