<?php
/**
 * Génération de PDF pour les devis
 *
 * @package WC_Quotes_Woodmart
 */

if (!defined('ABSPATH')) {
    exit;
}

class WCQ_PDF {
    
    /**
     * Constructeur
     */
    public function __construct() {
        // Hook pour le téléchargement de PDF
        add_action('admin_post_wcq_download_pdf', array($this, 'download_pdf'));
        add_action('admin_post_nopriv_wcq_download_pdf', array($this, 'download_pdf'));
        
        // AJAX pour générer et télécharger le PDF
        add_action('wp_ajax_wcq_generate_pdf', array($this, 'ajax_generate_pdf'));
        add_action('wp_ajax_nopriv_wcq_generate_pdf', array($this, 'ajax_generate_pdf'));
    }
    
    /**
     * Télécharger le PDF d'un devis
     */
    public function download_pdf() {
        // Vérifier le nonce
        if (!isset($_GET['_wpnonce']) || !wp_verify_nonce($_GET['_wpnonce'], 'wcq_download_pdf')) {
            wp_die(__('Action non autorisée', 'wc-quotes-woodmart'));
        }
        
        // Récupérer l'ID du devis
        $quote_id = isset($_GET['quote_id']) ? absint($_GET['quote_id']) : 0;
        
        if (!$quote_id) {
            wp_die(__('ID de devis invalide', 'wc-quotes-woodmart'));
        }
        
        // Vérifier les permissions
        $quote = WCQ_Quote::get_quote($quote_id);
        
        if (!$quote) {
            wp_die(__('Devis introuvable', 'wc-quotes-woodmart'));
        }
        
        // Vérifier que l'utilisateur a le droit d'accéder à ce devis
        if (!current_user_can('manage_woocommerce') && $quote['customer_id'] != get_current_user_id()) {
            wp_die(__('Vous n\'avez pas la permission d\'accéder à ce devis', 'wc-quotes-woodmart'));
        }
        
        // Générer le PDF
        self::generate_pdf($quote_id);
    }

    /**
     * AJAX : Générer et télécharger le PDF
     */
    public function ajax_generate_pdf() {
        check_ajax_referer('wcq_nonce', 'nonce');

        $quote_id = isset($_POST['quote_id']) ? absint($_POST['quote_id']) : 0;

        if (!$quote_id) {
            wp_send_json_error(array(
                'message' => __('ID de devis invalide', 'wc-quotes-woodmart')
            ));
        }

        self::generate_pdf($quote_id);
    }

    /**
     * Générer le PDF d'un devis
     *
     * @param int $quote_id ID du devis
     * @param bool $force_download (optionnel) Force le téléchargement (paramètre pour compatibilité)
     */
    public static function generate_pdf($quote_id, $force_download = true) {
        $quote = WCQ_Quote::get_quote($quote_id);
        
        if (!$quote) {
            wp_die(__('Devis introuvable', 'wc-quotes-woodmart'));
        }
        
        // Récupérer les informations du site
        $site_name = get_bloginfo('name');
        $site_url = get_bloginfo('url');
        
        // Récupérer le logo du site
        $logo_url = '';
        $custom_logo_id = get_theme_mod('custom_logo');
        if ($custom_logo_id) {
            $logo_url = wp_get_attachment_image_src($custom_logo_id, 'full')[0];
        }
        
        // Récupérer les informations du client
        $customer_name = '';
        $customer_email = '';
        if ($quote['customer_id'] > 0) {
            $user = get_userdata($quote['customer_id']);
            if ($user) {
                $customer_name = $user->display_name;
                $customer_email = $user->user_email;
            }
        }
        
        // Créer le contenu HTML du PDF
        ob_start();
        ?>
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset="UTF-8">
            <style>
                body {
                    font-family: Arial, sans-serif;
                    font-size: 12px;
                    line-height: 1.6;
                    color: #333;
                }
                .header {
                    margin-bottom: 30px;
                    padding-bottom: 20px;
                    border-bottom: 2px solid #000;
                }
                .logo {
                    max-width: 200px;
                    max-height: 80px;
                }
                .company-info {
                    margin-top: 10px;
                }
                .quote-info {
                    margin: 30px 0;
                    background: #f5f5f5;
                    padding: 15px;
                }
                .quote-info table {
                    width: 100%;
                }
                .quote-info td {
                    padding: 5px;
                }
                .customer-info {
                    margin-bottom: 30px;
                }
                table.items {
                    width: 100%;
                    border-collapse: collapse;
                    margin: 20px 0;
                }
                table.items th {
                    background: #333;
                    color: #fff;
                    padding: 10px;
                    text-align: left;
                    font-weight: bold;
                }
                table.items td {
                    padding: 10px;
                    border-bottom: 1px solid #ddd;
                }
                table.items tr:last-child td {
                    border-bottom: 2px solid #000;
                }
                .totals {
                    margin-top: 20px;
                    float: right;
                    width: 300px;
                }
                .totals table {
                    width: 100%;
                }
                .totals td {
                    padding: 8px;
                }
                .totals .total-row {
                    font-weight: bold;
                    font-size: 14px;
                    background: #f5f5f5;
                }
                .footer {
                    margin-top: 50px;
                    padding-top: 20px;
                    border-top: 1px solid #ddd;
                    text-align: center;
                    font-size: 10px;
                    color: #666;
                }
            </style>
        </head>
        <body>
            <div class="header">
                <?php if ($logo_url) : ?>
                    <img src="<?php echo esc_url($logo_url); ?>" alt="<?php echo esc_attr($site_name); ?>" class="logo">
                <?php endif; ?>
                <div class="company-info">
                    <strong><?php echo esc_html($site_name); ?></strong><br>
                    <?php echo esc_url($site_url); ?>
                </div>
            </div>
            
            <h1 style="margin-bottom: 20px;">Devis #<?php echo esc_html($quote['quote_number']); ?></h1>
            
            <div class="quote-info">
                <table>
                    <tr>
                        <td><strong>Date d'émission :</strong></td>
                        <td><?php echo date_i18n(get_option('date_format'), strtotime($quote['created_at'])); ?></td>
                    </tr>
                    <?php if ($quote['valid_until']) : ?>
                    <tr>
                        <td><strong>Valide jusqu'au :</strong></td>
                        <td><?php echo date_i18n(get_option('date_format'), strtotime($quote['valid_until'])); ?></td>
                    </tr>
                    <?php endif; ?>
                </table>
            </div>
            
            <div class="customer-info">
                <h3>Client</h3>
                <strong><?php echo esc_html($customer_name); ?></strong><br>
                <?php echo esc_html($customer_email); ?>
            </div>
            
            <table class="items">
                <thead>
                    <tr>
                        <th>Produit</th>
                        <th style="text-align: right;">Prix unitaire</th>
                        <th style="text-align: center;">Quantité</th>
                        <th style="text-align: right;">Total</th>
                    </tr>
                </thead>
                <tbody>
                    <?php 
                    foreach ($quote['quote_data'] as $item) :
                        $product_id = isset($item['variation_id']) && $item['variation_id'] > 0 ? $item['variation_id'] : $item['product_id'];
                        $product = wc_get_product($product_id);
                        
                        if (!$product) continue;
                        
                        $unit_price = $item['line_total'] / $item['quantity'];
                    ?>
                    <tr>
                        <td>
                            <?php echo esc_html($product->get_name()); ?>
                            <?php if (isset($item['variation']) && !empty($item['variation'])) : ?>
                                <br><small>
                                <?php 
                                $variations = array();
                                foreach ($item['variation'] as $key => $value) {
                                    $variations[] = wc_attribute_label($key) . ': ' . $value;
                                }
                                echo esc_html(implode(', ', $variations));
                                ?>
                                </small>
                            <?php endif; ?>
                        </td>
                        <td style="text-align: right;"><?php echo wc_price($unit_price); ?></td>
                        <td style="text-align: center;"><?php echo esc_html($item['quantity']); ?></td>
                        <td style="text-align: right;"><?php echo wc_price($item['line_total']); ?></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            
            <div class="totals">
                <table>
                    <tr class="total-row">
                        <td>Total TTC</td>
                        <td style="text-align: right;"><?php echo wc_price($quote['total']); ?></td>
                    </tr>
                </table>
            </div>
            
            <div style="clear: both;"></div>
            
            <div class="footer">
                <p>Ce devis est valable jusqu'à la date indiquée ci-dessus.</p>
                <p><?php echo esc_html($site_name); ?> - <?php echo esc_url($site_url); ?></p>
            </div>
        </body>
        </html>
        <?php
        $html = ob_get_clean();

        // Utiliser FPDF pour générer un vrai PDF
        self::generate_pdf_with_fpdf($quote_id);
    }

    /**
     * Obtenir le chemin local du logo du site
     *
     * @return string|false Chemin absolu vers le logo ou false si non disponible
     */
    private static function get_logo_path() {
        // Utiliser le logo local
        $logo_url = 'http://my-metrology-v3.local/wp-content/uploads/2025/05/logo.png';

        // Télécharger temporairement le logo pour FPDF
        $temp_file = tempnam(sys_get_temp_dir(), 'logo_') . '.png';

        $response = wp_remote_get($logo_url, array(
            'timeout' => 10,
            'sslverify' => false // Désactiver la vérification SSL pour les URLs locales
        ));

        if (!is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200) {
            $image_data = wp_remote_retrieve_body($response);
            file_put_contents($temp_file, $image_data);

            // Vérifier que le fichier existe
            if (file_exists($temp_file)) {
                return $temp_file;
            }
        }

        return false;
    }

    /**
     * Formater un prix pour FPDF (sans HTML)
     *
     * @param float $price Prix à formater
     * @return string Prix formaté
     */
    private static function format_price($price) {
        // Utiliser "euro" au lieu du symbole €
        return number_format($price, 2, ',', ' ') . ' euro';
    }

    /**
     * Générer le PDF avec FPDF
     *
     * @param int $quote_id ID du devis
     */
    private static function generate_pdf_with_fpdf($quote_id) {
        // Charger FPDF
        require_once WCQ_PLUGIN_DIR . 'includes/libraries/fpdf.php';

        $quote = WCQ_Quote::get_quote($quote_id);

        if (!$quote) {
            wp_die(__('Devis introuvable', 'wc-quotes-woodmart'));
        }

        // Créer le PDF
        $pdf = new \FPDF('P', 'mm', 'A4');
        $pdf->AddPage();
        $pdf->SetAutoPageBreak(true, 15);

        // En-tête - Logo et info site
        $logo_path = self::get_logo_path();
        $logo_displayed = false;

        if ($logo_path && file_exists($logo_path)) {
            // Vérifier l'extension du logo
            $logo_ext = strtolower(pathinfo($logo_path, PATHINFO_EXTENSION));
            if (in_array($logo_ext, array('jpg', 'jpeg', 'png', 'webp'))) {
                try {
                    // Tenter d'afficher le logo
                    $pdf->Image($logo_path, 10, 10, 50);
                    $logo_displayed = true;
                } catch (Exception $e) {
                    // Si l'image ne peut pas être chargée, on continue sans
                    $logo_displayed = false;
                }
            }
        }

        // Si le logo est affiché, décaler le contenu
        if ($logo_displayed) {
            $pdf->SetY(40);
        }

        $pdf->SetFont('Arial', 'B', 12);
        $pdf->Cell(0, 6, utf8_decode(get_bloginfo('name')), 0, 1);
        $pdf->SetFont('Arial', '', 10);
        $pdf->Cell(0, 5, utf8_decode(get_bloginfo('url')), 0, 1);

        $pdf->Ln(5);

        // Ligne de séparation
        $pdf->SetDrawColor(0, 0, 0);
        $pdf->SetLineWidth(0.5);
        $pdf->Line(10, $pdf->GetY(), 200, $pdf->GetY());
        $pdf->Ln(10);

        // Titre du devis
        $pdf->SetFont('Arial', 'B', 20);
        $pdf->Cell(0, 10, utf8_decode('DEVIS #' . $quote['quote_number']), 0, 1, 'C');
        $pdf->Ln(5);

        // Informations du devis
        $pdf->SetFont('Arial', '', 11);
        $pdf->Cell(50, 7, utf8_decode('Date d\'émission :'), 0, 0);
        $pdf->SetFont('Arial', 'B', 11);
        $pdf->Cell(0, 7, utf8_decode(date_i18n(get_option('date_format'), strtotime($quote['created_at']))), 0, 1);

        if (!empty($quote['valid_until'])) {
            $pdf->SetFont('Arial', '', 11);
            $pdf->Cell(50, 7, utf8_decode('Valide jusqu\'au :'), 0, 0);
            $pdf->SetFont('Arial', 'B', 11);
            $pdf->Cell(0, 7, utf8_decode(date_i18n(get_option('date_format'), strtotime($quote['valid_until']))), 0, 1);
        }

        $pdf->Ln(5);

        // Informations client
        $pdf->SetFont('Arial', 'B', 12);
        $pdf->Cell(0, 7, utf8_decode('Client :'), 0, 1);

        if ($quote['customer_id'] > 0) {
            $user = get_userdata($quote['customer_id']);
            if ($user) {
                $pdf->SetFont('Arial', '', 11);
                $pdf->Cell(0, 6, utf8_decode($user->display_name), 0, 1);
                $pdf->Cell(0, 6, utf8_decode($user->user_email), 0, 1);
            }
        }

        $pdf->Ln(8);

        // Tableau des produits - En-tête
        $pdf->SetFont('Arial', 'B', 9);
        $pdf->SetFillColor(230, 230, 230);

        $pdf->Cell(10, 9, utf8_decode('N°'), 1, 0, 'C', true);
        $pdf->Cell(28, 9, utf8_decode('Référence'), 1, 0, 'L', true);
        $pdf->Cell(12, 9, utf8_decode('Qté'), 1, 0, 'C', true);
        $pdf->Cell(30, 9, utf8_decode('PU HT avant'), 1, 0, 'R', true);
        $pdf->Cell(18, 9, utf8_decode('Remise'), 1, 0, 'C', true);
        $pdf->Cell(30, 9, utf8_decode('PU HT après'), 1, 0, 'R', true);
        $pdf->Cell(37, 9, utf8_decode('Total HT'), 1, 1, 'R', true);

        // Lignes de produits
        $pdf->SetFont('Arial', '', 9);
        $line_number = 1;
        foreach ($quote['quote_data'] as $item) {
            $product_id = isset($item['variation_id']) && $item['variation_id'] > 0 ? $item['variation_id'] : $item['product_id'];
            $product = wc_get_product($product_id);

            if (!$product) {
                continue;
            }

            // Calculer la remise
            $line_subtotal = isset($item['line_subtotal']) ? floatval($item['line_subtotal']) : floatval($item['line_total']);
            $line_total = floatval($item['line_total']);
            $discount_percent = 0;
            $has_discount = false;

            if ($line_subtotal > $line_total && $line_subtotal > 0) {
                $discount_percent = round((($line_subtotal - $line_total) / $line_subtotal) * 100);
                $has_discount = true;
            }

            $unit_price_before = $line_subtotal / $item['quantity'];
            $unit_price_after = $line_total / $item['quantity'];

            // Construire les informations du produit pour l'affichage détaillé
            $product_name = $product->get_name();
            $product_details = array();

            // Ajouter les variations
            if (isset($item['variation']) && !empty($item['variation'])) {
                $variations = array();
                foreach ($item['variation'] as $key => $value) {
                    $clean_key = str_replace('attribute_', '', $key);
                    $label = wc_attribute_label($clean_key);
                    $term = get_term_by('slug', $value, $clean_key);
                    $display_value = $term ? $term->name : $value;
                    $variations[] = $label . ': ' . $display_value;
                }
                if (!empty($variations)) {
                    $product_details[] = '(' . implode(', ', $variations) . ')';
                }
            }

            // Ajouter la date de livraison
            foreach ($item as $key => $value) {
                if (in_array($key, array('product_id', 'variation_id', 'variation', 'quantity', 'line_total', 'line_subtotal', 'line_tax', 'line_subtotal_tax', 'data', 'data_hash', 'key'))) {
                    continue;
                }
                $is_delivery = (stripos($key, 'livraison') !== false || stripos($key, 'delivery') !== false);
                if ($is_delivery && !empty($value)) {
                    $delivery_text = '';
                    if (is_array($value)) {
                        if (isset($value['value'])) {
                            $delivery_text = strip_tags($value['value']);
                        } elseif (isset($value[0])) {
                            $delivery_text = strip_tags($value[0]);
                        }
                    } elseif (is_object($value)) {
                        if (isset($value->value)) {
                            $delivery_text = strip_tags($value->value);
                        }
                    } else {
                        $delivery_text = strip_tags($value);
                    }
                    if (!empty($delivery_text)) {
                        $product_details[] = 'Livraison estimee: ' . $delivery_text;
                    }
                    break;
                }
            }

            // Récupérer le SKU
            $sku = $product->get_sku();
            if (empty($sku)) {
                $sku = '-';
            }

            $y_start = $pdf->GetY();

            // Si il y a des détails (variations ou livraison), afficher le nom du produit sur une ligne séparée
            if (!empty($product_details)) {
                $pdf->SetFont('Arial', 'B', 9);
                $pdf->Cell(165, 5, utf8_decode($product_name), 1, 1, 'L');
                $pdf->SetFont('Arial', '', 8);
                $pdf->Cell(165, 4, utf8_decode(implode(' - ', $product_details)), 1, 1, 'L');
                $pdf->SetFont('Arial', '', 9);
            }

            // Ligne avec les données chiffrées
            $pdf->Cell(10, 7, $line_number, 1, 0, 'C');
            $pdf->Cell(28, 7, utf8_decode($sku), 1, 0, 'L');
            $pdf->Cell(12, 7, $item['quantity'], 1, 0, 'C');
            $pdf->Cell(30, 7, utf8_decode(self::format_price($unit_price_before)), 1, 0, 'R');
            $pdf->Cell(18, 7, $discount_percent > 0 ? $discount_percent . '%' : '-', 1, 0, 'C');
            $pdf->Cell(30, 7, utf8_decode(self::format_price($unit_price_after)), 1, 0, 'R');
            $pdf->Cell(37, 7, utf8_decode(self::format_price($line_total)), 1, 1, 'R');

            $line_number++;
        }

        // Totaux
        $pdf->Ln(5);

        // Calculer le sous-total (somme des lignes de produits)
        $subtotal = 0;
        foreach ($quote['quote_data'] as $item) {
            $subtotal += $item['line_total'];
        }

        $shipping_cost = !empty($quote['shipping_cost']) ? floatval($quote['shipping_cost']) : 0;

        // Afficher le sous-total
        $pdf->SetFont('Arial', '', 11);
        $pdf->Cell(110, 7, '', 0, 0);
        $pdf->Cell(35, 7, utf8_decode('Sous-total HT :'), 1, 0, 'L');
        $pdf->Cell(50, 7, utf8_decode(self::format_price($subtotal)), 1, 1, 'R');

        // Afficher les frais de port s'ils existent
        if ($shipping_cost > 0) {
            $pdf->Cell(110, 7, '', 0, 0);
            $pdf->Cell(35, 7, utf8_decode('Frais de port :'), 1, 0, 'L');
            $pdf->Cell(50, 7, utf8_decode(self::format_price($shipping_cost)), 1, 1, 'R');
        }

        // Afficher le total
        $pdf->SetFont('Arial', 'B', 13);
        $pdf->SetFillColor(230, 230, 230);
        $pdf->Cell(110, 9, '', 0, 0);
        $pdf->Cell(35, 9, utf8_decode('TOTAL TTC :'), 1, 0, 'L', true);
        $pdf->Cell(50, 9, utf8_decode(self::format_price($quote['total'])), 1, 1, 'R', true);

        // Footer
        $pdf->Ln(10);
        $pdf->SetFont('Arial', 'I', 9);
        $pdf->MultiCell(0, 5, utf8_decode('Ce devis est valable jusqu\'à la date indiquée ci-dessus.'), 0, 'C');

        $pdf->Ln(3);
        $pdf->SetFont('Arial', '', 9);
        $pdf->Cell(0, 5, utf8_decode(get_bloginfo('name') . ' - ' . get_bloginfo('url')), 0, 1, 'C');

        // Output
        $pdf->Output('D', 'devis-' . $quote['quote_number'] . '.pdf');
        exit;
    }

    /**
     * Créer le PDF avec DomPDF (OBSOLÈTE - gardé pour compatibilité)
     */
    private static function create_pdf_with_dompdf($html, $quote_number) {
        // Vérifier si DomPDF est disponible
        if (!class_exists('Dompdf\Dompdf')) {
            // Si DomPDF n'est pas disponible, créer un PDF simple
            self::create_simple_pdf($html, $quote_number);
            return;
        }

        require_once WCQ_PLUGIN_DIR . 'vendor/autoload.php';

        $dompdf = new \Dompdf\Dompdf();
        $dompdf->loadHtml($html);
        $dompdf->setPaper('A4', 'portrait');
        $dompdf->render();

        // Envoyer le PDF au navigateur
        $dompdf->stream('devis-' . $quote_number . '.pdf', array('Attachment' => 1));
        exit;
    }

    /**
     * Créer un PDF simple sans bibliothèque externe
     */
    private static function create_simple_pdf($html, $quote_number) {
        // Pour une version basique, on va simplement convertir le HTML en PDF
        // en utilisant wkhtmltopdf si disponible, sinon on retourne le HTML
        
        header('Content-Type: application/pdf');
        header('Content-Disposition: attachment; filename="devis-' . $quote_number . '.pdf"');
        header('Cache-Control: private, max-age=0, must-revalidate');
        header('Pragma: public');
        
        // Version simplifiée : convertir HTML en PDF basique
        // Dans un environnement de production, utilisez une vraie bibliothèque PDF
        echo $html;
        exit;
    }
}

new WCQ_PDF();
