<?php
/**
 * Gestion des devis
 *
 * @package WC_Quotes_Woodmart
 */

if (!defined('ABSPATH')) {
    exit;
}

class WCQ_Quote {
    
    /**
     * Constructeur
     */
    public function __construct() {
        // AJAX pour convertir un devis en commande
        add_action('wp_ajax_wcq_convert_to_order', array($this, 'ajax_convert_to_order'));
        
        // AJAX pour supprimer un devis
        add_action('wp_ajax_wcq_delete_quote', array($this, 'ajax_delete_quote'));
    }
    
    /**
     * Créer un nouveau devis
     */
    public static function create_quote($cart_data, $user_id = 0, $valid_days = 30, $args = array()) {
        // Si pas d'utilisateur spécifié, utiliser l'utilisateur courant
        if ($user_id === 0) {
            $user_id = get_current_user_id();
        }

        // Générer un numéro de devis unique
        $quote_number = self::generate_quote_number();

        // Calculer le total
        $total = 0;
        foreach ($cart_data as $cart_item) {
            $total += $cart_item['line_total'];
        }

        // Ajouter les frais de port au total si présents
        $shipping_cost = isset($args['shipping_cost']) ? floatval($args['shipping_cost']) : 0;
        $total += $shipping_cost;

        // Calculer la date de validité
        $valid_until = null;
        if ($valid_days > 0) {
            $valid_until = date('Y-m-d H:i:s', strtotime("+{$valid_days} days"));
        }

        // Créer le post devis
        $quote_id = wp_insert_post(array(
            'post_type'   => 'shop_quote',
            'post_status' => 'quote-pending',
            'post_title'  => sprintf(__('Devis #%s', 'wc-quotes-woodmart'), $quote_number),
            'post_author' => $user_id,
        ));

        if (is_wp_error($quote_id)) {
            return false;
        }

        // Sauvegarder les métadonnées
        update_post_meta($quote_id, '_quote_number', $quote_number);
        update_post_meta($quote_id, '_quote_data', $cart_data);
        update_post_meta($quote_id, '_quote_total', $total);
        update_post_meta($quote_id, '_valid_until', $valid_until);
        update_post_meta($quote_id, '_customer_id', $user_id);
        update_post_meta($quote_id, '_is_read', '0'); // Marquer comme non lu par défaut

        // Sauvegarder les frais de port
        if ($shipping_cost > 0) {
            update_post_meta($quote_id, '_shipping_cost', $shipping_cost);
        }

        // Sauvegarder les informations client
        if ($user_id > 0) {
            $user = get_userdata($user_id);
            update_post_meta($quote_id, '_customer_email', $user->user_email);
            update_post_meta($quote_id, '_customer_first_name', get_user_meta($user_id, 'first_name', true));
            update_post_meta($quote_id, '_customer_last_name', get_user_meta($user_id, 'last_name', true));
        }

        return $quote_id;
    }
    
    /**
     * Générer un numéro de devis unique
     */
    private static function generate_quote_number() {
        $prefix = apply_filters('wcq_quote_number_prefix', 'DEV-');
        $number = get_option('wcq_last_quote_number', 0) + 1;
        update_option('wcq_last_quote_number', $number);
        
        return $prefix . str_pad($number, 6, '0', STR_PAD_LEFT);
    }
    
    /**
     * Obtenir un devis par son ID
     */
    public static function get_quote($quote_id) {
        $quote = get_post($quote_id);

        if (!$quote || $quote->post_type !== 'shop_quote') {
            return false;
        }

        return array(
            'id'            => $quote->ID,
            'quote_number'  => get_post_meta($quote->ID, '_quote_number', true),
            'quote_data'    => get_post_meta($quote->ID, '_quote_data', true),
            'total'         => get_post_meta($quote->ID, '_quote_total', true),
            'status'        => $quote->post_status,
            'valid_until'   => get_post_meta($quote->ID, '_valid_until', true),
            'customer_id'   => get_post_meta($quote->ID, '_customer_id', true),
            'created_at'    => $quote->post_date,
            'modified_at'   => $quote->post_modified,
            'shipping_cost' => get_post_meta($quote->ID, '_shipping_cost', true),
        );
    }
    
    /**
     * Obtenir tous les devis d'un utilisateur
     */
    public static function get_user_quotes($user_id) {
        $args = array(
            'post_type'      => 'shop_quote',
            'post_status'    => array('quote-pending', 'quote-accepted', 'quote-rejected', 'quote-expired', 'quote-replaced', 'quote-transferred'),
            'posts_per_page' => -1,
            'author'         => $user_id,
            'orderby'        => 'date',
            'order'          => 'DESC',
        );

        $quotes = get_posts($args);
        $result = array();

        foreach ($quotes as $quote) {
            $result[] = self::get_quote($quote->ID);
        }

        return $result;
    }
    
    /**
     * Convertir un devis en commande WooCommerce
     */
    public static function convert_to_order($quote_id) {
        $quote = self::get_quote($quote_id);
        
        if (!$quote) {
            return new WP_Error('invalid_quote', __('Devis invalide', 'wc-quotes-woodmart'));
        }
        
        // Créer une nouvelle commande
        $order = wc_create_order(array(
            'customer_id' => $quote['customer_id'],
        ));
        
        if (is_wp_error($order)) {
            return $order;
        }
        
        // Ajouter les produits du devis à la commande
        foreach ($quote['quote_data'] as $cart_item) {
            $product_id   = $cart_item['product_id'];
            $variation_id = isset($cart_item['variation_id']) ? $cart_item['variation_id'] : 0;
            $quantity     = $cart_item['quantity'];
            
            $order->add_product(
                wc_get_product($variation_id > 0 ? $variation_id : $product_id),
                $quantity
            );
        }
        
        // Calculer les totaux
        $order->calculate_totals();
        
        // Mettre à jour le statut du devis
        wp_update_post(array(
            'ID'          => $quote_id,
            'post_status' => 'quote-accepted',
        ));
        
        // Ajouter une note à la commande
        $order->add_order_note(
            sprintf(__('Commande créée depuis le devis #%s', 'wc-quotes-woodmart'), $quote['quote_number'])
        );
        
        // Sauvegarder la référence du devis dans la commande
        update_post_meta($order->get_id(), '_created_from_quote', $quote_id);
        update_post_meta($quote_id, '_converted_order_id', $order->get_id());
        
        return $order->get_id();
    }
    
    /**
     * AJAX : Convertir un devis en commande
     */
    public function ajax_convert_to_order() {
        check_ajax_referer('wcq_admin_nonce', 'nonce');
        
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(array(
                'message' => __('Permissions insuffisantes', 'wc-quotes-woodmart')
            ));
        }
        
        $quote_id = isset($_POST['quote_id']) ? intval($_POST['quote_id']) : 0;
        
        if (!$quote_id) {
            wp_send_json_error(array(
                'message' => __('ID de devis invalide', 'wc-quotes-woodmart')
            ));
        }
        
        $order_id = self::convert_to_order($quote_id);
        
        if (is_wp_error($order_id)) {
            wp_send_json_error(array(
                'message' => $order_id->get_error_message()
            ));
        }
        
        wp_send_json_success(array(
            'message'  => __('Devis converti en commande avec succès', 'wc-quotes-woodmart'),
            'order_id' => $order_id,
            'redirect' => admin_url('post.php?post=' . $order_id . '&action=edit')
        ));
    }
    
    /**
     * AJAX : Supprimer un devis
     */
    public function ajax_delete_quote() {
        check_ajax_referer('wcq_nonce', 'nonce');
        
        $quote_id = isset($_POST['quote_id']) ? intval($_POST['quote_id']) : 0;
        
        if (!$quote_id) {
            wp_send_json_error(array(
                'message' => __('ID de devis invalide', 'wc-quotes-woodmart')
            ));
        }
        
        // Vérifier que l'utilisateur a le droit de supprimer ce devis
        $quote = self::get_quote($quote_id);
        if ($quote['customer_id'] != get_current_user_id() && !current_user_can('manage_woocommerce')) {
            wp_send_json_error(array(
                'message' => __('Permissions insuffisantes', 'wc-quotes-woodmart')
            ));
        }
        
        $deleted = wp_delete_post($quote_id, true);
        
        if (!$deleted) {
            wp_send_json_error(array(
                'message' => __('Erreur lors de la suppression du devis', 'wc-quotes-woodmart')
            ));
        }
        
        wp_send_json_success(array(
            'message' => __('Devis supprimé avec succès', 'wc-quotes-woodmart')
        ));
    }
}

new WCQ_Quote();
