<?php
/**
 * Template Name: Édition de Devis
 * Page panier dédiée à l'édition de devis
 *
 * @package WC_Quotes_Woodmart
 */

if (!defined('ABSPATH')) {
    exit;
}

// Vérifier que l'utilisateur est connecté
if (!is_user_logged_in()) {
    wp_redirect(wp_login_url(get_permalink()));
    exit;
}

// Vérifier qu'on a un quote_id
$quote_id = isset($_GET['quote_id']) ? intval($_GET['quote_id']) : 0;

if (!$quote_id) {
    wc_add_notice(__('Aucun devis spécifié.', 'wc-quotes-woodmart'), 'error');
    wp_redirect(wc_get_account_endpoint_url('quotes'));
    exit;
}

// Vérifier que le devis existe et appartient au client
$quote = get_post($quote_id);
if (!$quote || $quote->post_type !== 'shop_quote') {
    wc_add_notice(__('Devis introuvable.', 'wc-quotes-woodmart'), 'error');
    wp_redirect(wc_get_account_endpoint_url('quotes'));
    exit;
}

$customer_id = get_post_meta($quote_id, '_customer_id', true);
if ($customer_id != get_current_user_id()) {
    wc_add_notice(__('Vous n\'avez pas accès à ce devis.', 'wc-quotes-woodmart'), 'error');
    wp_redirect(wc_get_account_endpoint_url('quotes'));
    exit;
}

// Charger le devis dans le panier si ce n'est pas déjà fait
if (WC()->cart->is_empty() || !WC()->session->get('editing_quote_id')) {
    WC()->cart->empty_cart();

    $quote_data = get_post_meta($quote_id, '_quote_data', true);

    if (!empty($quote_data)) {
        foreach ($quote_data as $item) {
            $product_id = $item['product_id'];
            $quantity = $item['quantity'];
            $variation_id = isset($item['variation_id']) ? $item['variation_id'] : 0;
            $variation = isset($item['variation']) ? $item['variation'] : array();

            WC()->cart->add_to_cart($product_id, $quantity, $variation_id, $variation);
        }
    }

    // Marquer qu'on est en mode édition
    WC()->session->set('editing_quote_id', $quote_id);
}

// Retirer les boutons "Transformer en devis" et "Valider la commande" sur cette page
remove_action('woocommerce_proceed_to_checkout', 'woocommerce_button_proceed_to_checkout', 20);
add_action('woocommerce_proceed_to_checkout', function() {
    // Ne rien afficher - on masque les boutons de checkout sur la page d'édition
}, 1);

get_header('shop');
?>

<div class="woocommerce">
    <div class="container">
        
        <div class="wcq-edit-quote-header">
            <h1><?php echo sprintf(__('Modifier le devis %s', 'wc-quotes-woodmart'), get_post_meta($quote_id, '_quote_number', true)); ?></h1>
            <p class="wcq-edit-notice">
                <?php _e('Modifiez les quantités, ajoutez ou supprimez des produits, puis cliquez sur "Mettre à jour le devis".', 'wc-quotes-woodmart'); ?>
            </p>
            <p>
                <a href="<?php echo esc_url(wc_get_account_endpoint_url('quotes') . $quote_id . '/'); ?>" class="button">
                    <?php _e('← Annuler et retourner au devis', 'wc-quotes-woodmart'); ?>
                </a>
            </p>
        </div>

        <?php do_action('woocommerce_before_cart'); ?>

        <form class="woocommerce-cart-form" action="" method="post">
            <?php do_action('woocommerce_before_cart_table'); ?>

            <table class="shop_table shop_table_responsive shop-table-with-img cart woocommerce-cart-form__contents" cellspacing="0">
                <thead>
                    <tr>
                        <th class="product-remove"><span class="screen-reader-text"><?php esc_html_e('Remove item', 'woocommerce'); ?></span></th>
                        <th class="product-thumbnail"><span class="screen-reader-text"><?php esc_html_e('Thumbnail', 'woocommerce'); ?></span></th>
                        <th class="product-name"><?php esc_html_e('Product', 'woocommerce'); ?></th>
                        <th class="product-price"><?php esc_html_e('Price', 'woocommerce'); ?></th>
                        <th class="product-quantity"><?php esc_html_e('Quantity', 'woocommerce'); ?></th>
                        <th class="product-subtotal"><?php esc_html_e('Subtotal', 'woocommerce'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php do_action('woocommerce_before_cart_contents'); ?>

                    <?php
                    foreach (WC()->cart->get_cart() as $cart_item_key => $cart_item) {
                        $_product   = apply_filters('woocommerce_cart_item_product', $cart_item['data'], $cart_item, $cart_item_key);
                        $product_id = apply_filters('woocommerce_cart_item_product_id', $cart_item['product_id'], $cart_item, $cart_item_key);

                        if ($_product && $_product->exists() && $cart_item['quantity'] > 0 && apply_filters('woocommerce_cart_item_visible', true, $cart_item, $cart_item_key)) {
                            $product_permalink = apply_filters('woocommerce_cart_item_permalink', $_product->is_visible() ? $_product->get_permalink($cart_item) : '', $cart_item, $cart_item_key);
                            ?>
                            <tr class="woocommerce-cart-form__cart-item <?php echo esc_attr(apply_filters('woocommerce_cart_item_class', 'cart_item', $cart_item, $cart_item_key)); ?>">

                                <td class="product-remove">
                                    <?php
                                    echo apply_filters(
                                        'woocommerce_cart_item_remove_link',
                                        sprintf(
                                            '<a href="%s" class="remove" aria-label="%s" data-product_id="%s" data-product_sku="%s">&times;</a>',
                                            esc_url(wc_get_cart_remove_url($cart_item_key)),
                                            esc_attr__('Remove this item', 'woocommerce'),
                                            esc_attr($product_id),
                                            esc_attr($_product->get_sku())
                                        ),
                                        $cart_item_key
                                    );
                                    ?>
                                </td>

                                <td class="product-thumbnail">
                                <?php
                                $thumbnail = apply_filters('woocommerce_cart_item_thumbnail', $_product->get_image(), $cart_item, $cart_item_key);

                                if (!$product_permalink) {
                                    echo $thumbnail;
                                } else {
                                    printf('<a href="%s">%s</a>', esc_url($product_permalink), $thumbnail);
                                }
                                ?>
                                </td>

                                <td class="product-name" data-title="<?php esc_attr_e('Product', 'woocommerce'); ?>">
                                <?php
                                if (!$product_permalink) {
                                    echo wp_kses_post(apply_filters('woocommerce_cart_item_name', $_product->get_name(), $cart_item, $cart_item_key) . '&nbsp;');
                                } else {
                                    echo wp_kses_post(apply_filters('woocommerce_cart_item_name', sprintf('<a href="%s">%s</a>', esc_url($product_permalink), $_product->get_name()), $cart_item, $cart_item_key));
                                }

                                do_action('woocommerce_after_cart_item_name', $cart_item, $cart_item_key);

                                // SKU
                                if ($_product->get_sku()) {
                                    echo '<div class="wd-product-detail wd-product-sku">';
                                    echo '<span class="wd-label">' . esc_html__('SKU:', 'woocommerce') . '</span>';
                                    echo '<span>' . esc_html($_product->get_sku()) . '</span>';
                                    echo '</div>';
                                }

                                // Meta data
                                echo wc_get_formatted_cart_item_data($cart_item);

                                // Backorder notification
                                if ($_product->backorders_require_notification() && $_product->is_on_backorder($cart_item['quantity'])) {
                                    echo wp_kses_post(apply_filters('woocommerce_cart_item_backorder_notification', '<p class="backorder_notification">' . esc_html__('Available on backorder', 'woocommerce') . '</p>', $product_id));
                                }
                                ?>
                                </td>

                                <td class="product-price" data-title="<?php esc_attr_e('Price', 'woocommerce'); ?>">
                                    <?php
                                    echo apply_filters('woocommerce_cart_item_price', WC()->cart->get_product_price($_product), $cart_item, $cart_item_key);
                                    ?>
                                </td>

                                <td class="product-quantity" data-title="<?php esc_attr_e('Quantity', 'woocommerce'); ?>">
                                <?php
                                if ($_product->is_sold_individually()) {
                                    $product_quantity = sprintf('1 <input type="hidden" name="cart[%s][qty]" value="1" />', $cart_item_key);
                                } else {
                                    $product_quantity = woocommerce_quantity_input(
                                        array(
                                            'input_name'   => "cart[{$cart_item_key}][qty]",
                                            'input_value'  => $cart_item['quantity'],
                                            'max_value'    => $_product->get_max_purchase_quantity(),
                                            'min_value'    => '0',
                                            'product_name' => $_product->get_name(),
                                        ),
                                        $_product,
                                        false
                                    );
                                }

                                echo apply_filters('woocommerce_cart_item_quantity', $product_quantity, $cart_item_key, $cart_item);
                                ?>
                                </td>

                                <td class="product-subtotal" data-title="<?php esc_attr_e('Subtotal', 'woocommerce'); ?>">
                                    <?php
                                    echo apply_filters('woocommerce_cart_item_subtotal', WC()->cart->get_product_subtotal($_product, $cart_item['quantity']), $cart_item, $cart_item_key);
                                    ?>
                                </td>
                            </tr>
                            <?php
                        }
                    }
                    ?>

                    <?php do_action('woocommerce_cart_contents'); ?>

                    <tr>
                        <td colspan="6" class="actions">
                            <div class="cart-actions">
                                <button type="submit" class="button" name="update_cart" value="<?php esc_attr_e('Mettre à jour les quantités', 'wc-quotes-woodmart'); ?>">
                                    <?php esc_html_e('Mettre à jour les quantités', 'wc-quotes-woodmart'); ?>
                                </button>

                                <?php do_action('woocommerce_cart_actions'); ?>

                                <?php wp_nonce_field('woocommerce-cart', 'woocommerce-cart-nonce'); ?>
                            </div>
                        </td>
                    </tr>

                    <?php do_action('woocommerce_after_cart_contents'); ?>
                </tbody>
            </table>
            <?php do_action('woocommerce_after_cart_table'); ?>
        </form>

        <?php do_action('woocommerce_before_cart_collaterals'); ?>

        <div class="cart-collaterals">
            <?php
                /**
                 * Cart collaterals hook.
                 *
                 * @hooked woocommerce_cross_sell_display
                 * @hooked woocommerce_cart_totals - 10
                 */
                do_action('woocommerce_cart_collaterals');
            ?>
            
            <div class="wcq-update-quote-actions">
                <a href="<?php echo esc_url(add_query_arg(array('wcq_action' => 'update_from_cart', 'quote_id' => $quote_id, '_wpnonce' => wp_create_nonce('wcq_update_' . $quote_id)))); ?>" 
                   class="button alt wcq-update-quote-btn">
                    <?php _e('Mettre à jour le devis', 'wc-quotes-woodmart'); ?>
                </a>
                <p class="wcq-help-text">
                    <?php _e('Le devis sera mis à jour avec le contenu actuel du panier.', 'wc-quotes-woodmart'); ?>
                </p>
            </div>
        </div>

        <?php do_action('woocommerce_after_cart'); ?>

    </div>
</div>

<style>
/* Masquer les cross-sells uniquement sur la page d'édition de devis */
body.page-template-edit-quote .cross-sells {
    display: none !important;
}

/* Masquer les boutons "Transformer en devis" et "Valider la commande" sur la page d'édition de devis */
.wcq-edit-quote-header ~ * .wc-proceed-to-checkout,
.cart-collaterals .wc-proceed-to-checkout {
    display: none !important;
}

/* Style pour le bouton "Mettre à jour le devis" - couleur rouge personnalisée */
.wcq-update-quote-btn {
    background-color: #c00000 !important;
    border-color: #c00000 !important;
    color: #ffffff !important;
}

.wcq-update-quote-btn:hover {
    background-color: #a00000 !important;
    border-color: #900000 !important;
}

/* Style pour le bouton "Mettre à jour le devis" en rouge après modification */
.wcq-update-quote-btn.wcq-quantities-changed {
    background-color: #c00000 !important;
    border-color: #c00000 !important;
    color: #ffffff !important;
    animation: pulse-red 2s infinite;
}

.wcq-update-quote-btn.wcq-quantities-changed:hover {
    background-color: #a00000 !important;
    border-color: #900000 !important;
}

@keyframes pulse-red {
    0%, 100% {
        box-shadow: 0 0 0 0 rgba(192, 0, 0, 0.7);
    }
    50% {
        box-shadow: 0 0 0 10px rgba(192, 0, 0, 0);
    }
}
</style>

<script type="text/javascript">
jQuery(document).ready(function($) {
    // Stocker les quantités initiales
    var initialQuantities = {};

    $('.product-quantity input[type="number"]').each(function() {
        var cartItemKey = $(this).attr('name').match(/cart\[(.*?)\]/)[1];
        initialQuantities[cartItemKey] = $(this).val();
    });

    // Détecter le clic sur "Mettre à jour les quantités"
    $('button[name="update_cart"]').on('click', function(e) {
        // Vérifier si des quantités ont changé
        var hasChanges = false;

        $('.product-quantity input[type="number"]').each(function() {
            var cartItemKey = $(this).attr('name').match(/cart\[(.*?)\]/)[1];
            if (initialQuantities[cartItemKey] != $(this).val()) {
                hasChanges = true;
            }
        });

        if (hasChanges) {
            // Marquer que des changements ont été faits
            sessionStorage.setItem('wcq_quantities_changed_<?php echo $quote_id; ?>', '1');
        }
    });

    // Vérifier si des quantités ont été modifiées et changer la couleur du bouton
    if (sessionStorage.getItem('wcq_quantities_changed_<?php echo $quote_id; ?>') === '1') {
        $('.wcq-update-quote-btn').addClass('wcq-quantities-changed');

        // Mettre à jour le texte d'aide
        $('.wcq-help-text').html('<strong style="color: #dc3545;"><?php _e('⚠️ Vous avez modifié les quantités. Cliquez sur le bouton ci-dessus pour enregistrer les modifications dans le devis.', 'wc-quotes-woodmart'); ?></strong>');
    }

    // Nettoyer le flag après la mise à jour du devis
    $('.wcq-update-quote-btn').on('click', function() {
        sessionStorage.removeItem('wcq_quantities_changed_<?php echo $quote_id; ?>');
    });

    // Détecter les changements en temps réel
    $('.product-quantity input[type="number"]').on('change', function() {
        var hasChanges = false;

        $('.product-quantity input[type="number"]').each(function() {
            var cartItemKey = $(this).attr('name').match(/cart\[(.*?)\]/)[1];
            if (initialQuantities[cartItemKey] != $(this).val()) {
                hasChanges = true;
            }
        });

        if (hasChanges) {
            $('.wcq-update-quote-btn').addClass('wcq-quantities-changed');
            $('.wcq-help-text').html('<strong style="color: #dc3545;"><?php _e('⚠️ Vous avez modifié les quantités. N\'oubliez pas de mettre à jour le panier puis le devis.', 'wc-quotes-woodmart'); ?></strong>');
        } else {
            $('.wcq-update-quote-btn').removeClass('wcq-quantities-changed');
            $('.wcq-help-text').html('<?php _e('Le devis sera mis à jour avec le contenu actuel du panier.', 'wc-quotes-woodmart'); ?>');
        }
    });
});
</script>

<?php get_footer('shop'); ?>
