<?php
/**
 * Template d'affichage d'un devis unique
 *
 * @package WC_Quotes_Woodmart
 * @var array $quote Données du devis
 */

if (!defined('ABSPATH')) {
    exit;
}

$is_expired = false;
if ($quote['valid_until']) {
    $valid_date = strtotime($quote['valid_until']);
    $is_expired = $valid_date < current_time('timestamp');
}

$converted_order_id = get_post_meta($quote['id'], '_converted_order_id', true);
?>

<div class="wcq-single-quote">
    
    <p class="wcq-back-link">
        <a href="<?php echo esc_url(wc_get_account_endpoint_url('quotes')); ?>">
            ← <?php _e('Retour à mes devis', 'wc-quotes-woodmart'); ?>
        </a>
    </p>
    
    <h2><?php printf(__('Devis #%s', 'wc-quotes-woodmart'), $quote['quote_number']); ?></h2>
    
    <div class="wcq-quote-header">
        <div class="wcq-quote-info">
            <table class="woocommerce-table shop_table">
                <tbody>
                    <tr>
                        <th><?php _e('Date d\'émission', 'wc-quotes-woodmart'); ?>:</th>
                        <td><?php echo date_i18n(get_option('date_format'), strtotime($quote['created_at'])); ?></td>
                    </tr>
                    <tr>
                        <th><?php _e('Statut', 'wc-quotes-woodmart'); ?>:</th>
                        <td>
                            <?php
                            $status_labels = array(
                                'quote-pending'  => __('En attente', 'wc-quotes-woodmart'),
                                'quote-accepted' => __('Accepté', 'wc-quotes-woodmart'),
                                'quote-rejected' => __('Refusé', 'wc-quotes-woodmart'),
                                'quote-expired'  => __('Expiré', 'wc-quotes-woodmart'),
                            );
                            $status_class = str_replace('quote-', '', $quote['status']);
                            ?>
                            <span class="wcq-status wcq-status-<?php echo esc_attr($status_class); ?>">
                                <?php echo esc_html($status_labels[$quote['status']] ?? $quote['status']); ?>
                            </span>
                        </td>
                    </tr>
                    <?php if ($quote['valid_until']) : ?>
                    <tr>
                        <th><?php _e('Valide jusqu\'au', 'wc-quotes-woodmart'); ?>:</th>
                        <td class="<?php echo $is_expired ? 'wcq-expired' : ''; ?>">
                            <?php echo date_i18n(get_option('date_format'), strtotime($quote['valid_until'])); ?>
                            <?php if ($is_expired) : ?>
                                <br><span style="color: red;"><?php _e('Ce devis a expiré', 'wc-quotes-woodmart'); ?></span>
                            <?php endif; ?>
                        </td>
                    </tr>
                    <?php endif; ?>
                    <?php if ($converted_order_id) : ?>
                    <tr>
                        <th><?php _e('Commande créée', 'wc-quotes-woodmart'); ?>:</th>
                        <td>
                            <a href="<?php echo esc_url(wc_get_account_endpoint_url('view-order') . $converted_order_id . '/'); ?>">
                                #<?php echo $converted_order_id; ?>
                            </a>
                        </td>
                    </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
    
    <h3><?php _e('Produits', 'wc-quotes-woodmart'); ?></h3>
    
    <?php if ($quote['status'] === 'quote-pending') : ?>
        <p class="wcq-edit-notice">
            <?php _e('Vous pouvez modifier les quantités ou supprimer des articles de ce devis.', 'wc-quotes-woodmart'); ?>
        </p>
    <?php endif; ?>
    
    <form method="post" action="" class="wcq-update-quote-form">
        <?php wp_nonce_field('wcq_update_quote_' . $quote['id'], 'wcq_update_nonce'); ?>
        <input type="hidden" name="wcq_action" value="update_quote">
        <input type="hidden" name="quote_id" value="<?php echo esc_attr($quote['id']); ?>">
        
        <table class="woocommerce-table shop_table shop_table_responsive wcq-quote-items-table">
            <thead>
                <tr>
                    <th class="product-name"><?php _e('Produit', 'wc-quotes-woodmart'); ?></th>
                    <th class="product-price"><?php _e('Prix unitaire', 'wc-quotes-woodmart'); ?></th>
                    <th class="product-quantity"><?php _e('Quantité', 'wc-quotes-woodmart'); ?></th>
                    <th class="product-total"><?php _e('Total', 'wc-quotes-woodmart'); ?></th>
                    <?php if ($quote['status'] === 'quote-pending') : ?>
                        <th class="product-remove">&nbsp;</th>
                    <?php endif; ?>
                </tr>
            </thead>
            <tbody>
                <?php
                $item_index = 0;
                foreach ($quote['quote_data'] as $item) :
                    $product_id = isset($item['variation_id']) && $item['variation_id'] > 0 
                        ? $item['variation_id'] 
                        : $item['product_id'];
                    
                    $product = wc_get_product($product_id);
                    
                    if (!$product || !$product->exists()) {
                        continue;
                    }
                    
                    $product_permalink = $product->is_visible() ? $product->get_permalink() : '';
                    $unit_price = $item['line_total'] / $item['quantity'];
                ?>
                <tr class="wcq-quote-item" data-item-index="<?php echo esc_attr($item_index); ?>">
                    
                    <td class="product-name" data-title="<?php esc_attr_e('Produit', 'wc-quotes-woodmart'); ?>">
                        <div class="wcq-product-info">
                            <?php
                            $thumbnail = $product->get_image('thumbnail');
                            if ($thumbnail) {
                                echo '<div class="wcq-product-thumbnail">' . $thumbnail . '</div>';
                            }
                            ?>
                            <div class="wcq-product-details">
                                <?php
                                if (!$product_permalink) {
                                    echo '<strong>' . wp_kses_post($product->get_name()) . '</strong>';
                                } else {
                                    echo '<strong><a href="' . esc_url($product_permalink) . '">' . wp_kses_post($product->get_name()) . '</a></strong>';
                                }
                                ?>
                                
                                <?php
                                // Afficher le SKU (REF)
                                $sku = $product->get_sku();
                                if ($sku) {
                                    echo '<div class="wcq-product-sku"><strong>' . __('REF:', 'wc-quotes-woodmart') . '</strong> ' . esc_html($sku) . '</div>';
                                }
                                
                                // Afficher les variations/attributs
                                if (isset($item['variation']) && !empty($item['variation'])) {
                                    echo '<dl class="wcq-variation-details variation">';
                                    foreach ($item['variation'] as $key => $value) {
                                        $attribute_name = str_replace('attribute_', '', $key);
                                        $attribute_label = wc_attribute_label($attribute_name);
                                        
                                        echo '<dt class="variation-' . sanitize_html_class($attribute_name) . '">' . esc_html($attribute_label) . ':</dt>';
                                        echo '<dd class="variation-' . sanitize_html_class($attribute_name) . '">' . wp_kses_post($value) . '</dd>';
                                    }
                                    echo '</dl>';
                                }
                                
                                // Dimensions
                                if ($product->has_dimensions()) {
                                    $dimensions = $product->get_dimensions(false);
                                    if (!empty($dimensions['length']) || !empty($dimensions['width']) || !empty($dimensions['height'])) {
                                        echo '<div class="wcq-product-dimensions">';
                                        echo '<strong>' . __('Dimensions:', 'wc-quotes-woodmart') . '</strong> ';
                                        echo wc_format_dimensions($dimensions);
                                        echo '</div>';
                                    }
                                }
                                
                                // Poids
                                if ($product->has_weight()) {
                                    echo '<div class="wcq-product-weight">';
                                    echo '<strong>' . __('Poids:', 'wc-quotes-woodmart') . '</strong> ';
                                    echo wc_format_weight($product->get_weight());
                                    echo '</div>';
                                }
                                
                                // Métadonnées personnalisées
                                if (isset($item['item_meta']) && !empty($item['item_meta'])) {
                                    echo '<dl class="wcq-item-meta">';
                                    foreach ($item['item_meta'] as $meta_key => $meta_value) {
                                        if (is_array($meta_value)) {
                                            $meta_value = implode(', ', $meta_value);
                                        }
                                        echo '<dt>' . esc_html($meta_key) . ':</dt>';
                                        echo '<dd>' . wp_kses_post($meta_value) . '</dd>';
                                    }
                                    echo '</dl>';
                                }
                                ?>
                            </div>
                        </div>
                    </td>
                    
                    <td class="product-price" data-title="<?php esc_attr_e('Prix unitaire', 'wc-quotes-woodmart'); ?>">
                        <?php echo wc_price($unit_price); ?>
                        <input type="hidden" name="items[<?php echo esc_attr($item_index); ?>][unit_price]" value="<?php echo esc_attr($unit_price); ?>">
                        <input type="hidden" name="items[<?php echo esc_attr($item_index); ?>][product_id]" value="<?php echo esc_attr($item['product_id']); ?>">
                        <?php if (isset($item['variation_id'])) : ?>
                            <input type="hidden" name="items[<?php echo esc_attr($item_index); ?>][variation_id]" value="<?php echo esc_attr($item['variation_id']); ?>">
                        <?php endif; ?>
                        <?php if (isset($item['variation'])) : ?>
                            <input type="hidden" name="items[<?php echo esc_attr($item_index); ?>][variation]" value="<?php echo esc_attr(json_encode($item['variation'])); ?>">
                        <?php endif; ?>
                    </td>
                    
                    <td class="product-quantity" data-title="<?php esc_attr_e('Quantité', 'wc-quotes-woodmart'); ?>">
                        <?php if ($quote['status'] === 'quote-pending') : ?>
                            <div class="quantity">
                                <input 
                                    type="number" 
                                    name="items[<?php echo esc_attr($item_index); ?>][quantity]" 
                                    value="<?php echo esc_attr($item['quantity']); ?>" 
                                    min="1" 
                                    step="1"
                                    class="input-text qty text wcq-quantity-input"
                                    data-unit-price="<?php echo esc_attr($unit_price); ?>"
                                    data-item-index="<?php echo esc_attr($item_index); ?>"
                                >
                            </div>
                        <?php else : ?>
                            <?php echo esc_html($item['quantity']); ?>
                        <?php endif; ?>
                    </td>
                    
                    <td class="product-total" data-title="<?php esc_attr_e('Total', 'wc-quotes-woodmart'); ?>">
                        <span class="wcq-item-total" data-item-index="<?php echo esc_attr($item_index); ?>">
                            <?php echo wc_price($item['line_total']); ?>
                        </span>
                    </td>
                    
                    <?php if ($quote['status'] === 'quote-pending') : ?>
                        <td class="product-remove">
                            <button 
                                type="button" 
                                class="wcq-remove-item" 
                                data-item-index="<?php echo esc_attr($item_index); ?>"
                                title="<?php esc_attr_e('Supprimer cet article', 'wc-quotes-woodmart'); ?>"
                            >
                                <span class="dashicons dashicons-trash"></span>
                            </button>
                            <input type="hidden" name="items[<?php echo esc_attr($item_index); ?>][remove]" value="0" class="wcq-remove-flag">
                        </td>
                    <?php endif; ?>
                    
                </tr>
                <?php 
                    $item_index++;
                endforeach; 
                ?>
            </tbody>
            <tfoot>
                <tr class="order-total">
                    <th colspan="<?php echo $quote['status'] === 'quote-pending' ? '3' : '3'; ?>"><?php _e('Total', 'wc-quotes-woodmart'); ?></th>
                    <td colspan="<?php echo $quote['status'] === 'quote-pending' ? '2' : '1'; ?>">
                        <strong>
                            <span class="wcq-grand-total"><?php echo wc_price($quote['total']); ?></span>
                        </strong>
                    </td>
                </tr>
            </tfoot>
        </table>
        
        <?php if ($quote['status'] === 'quote-pending') : ?>
            <div class="wcq-actions">
                <button type="submit" class="button alt wcq-update-quote-btn">
                    <?php _e('Mettre à jour le devis', 'wc-quotes-woodmart'); ?>
                </button>
            </div>
        <?php endif; ?>
    </form>
    
    <div class="wcq-quote-actions">
        
        <?php if (!$converted_order_id && !$is_expired && $quote['status'] !== 'quote-rejected') : ?>
            <a href="<?php echo esc_url(wp_nonce_url(add_query_arg(array('convert-quote' => '1', 'quote_id' => $quote['id']), wc_get_account_endpoint_url('quotes')), 'convert_quote')); ?>" class="button alt">
                <?php _e('Commander maintenant', 'wc-quotes-woodmart'); ?>
            </a>
        <?php endif; ?>
        
        <a href="<?php echo esc_url(wp_nonce_url(admin_url('admin-post.php?action=wcq_download_pdf&quote_id=' . $quote['id']), 'wcq_download_pdf')); ?>" class="button" target="_blank">
            <?php _e('Télécharger en PDF', 'wc-quotes-woodmart'); ?>
        </a>
        
        <button type="button" class="button wcq-email-quote-btn" data-quote-id="<?php echo esc_attr($quote['id']); ?>">
            <?php _e('Recevoir par e-mail', 'wc-quotes-woodmart'); ?>
        </button>
        
        <?php if ($quote['status'] === 'quote-pending' && !$converted_order_id) : ?>
            <button type="button" class="button wcq-delete-quote-btn" data-quote-id="<?php echo esc_attr($quote['id']); ?>" style="float: right;">
                <?php _e('Supprimer', 'wc-quotes-woodmart'); ?>
            </button>
        <?php endif; ?>
        
    </div>
    
</div>
