<?php
/**
 * Plugin Name: WooCommerce Quotes for Woodmart
 * Plugin URI: https://github.com/yannvallet
 * Description: Plugin complet de gestion de devis WooCommerce compatible avec le thème Woodmart. Permet aux clients de créer des devis depuis le panier, les gérer dans leur espace client, et aux administrateurs de les convertir en commandes.
 * Version: 1.2.1
 * Author: Yann VALLET
 * Author URI: https://github.com/yannvallet
 * Text Domain: wc-quotes-woodmart
 * Domain Path: /languages
 * Requires at least: 6.0
 * Requires PHP: 8.0
 * Requires Plugins: woocommerce
 * WC requires at least: 7.0
 * WC tested up to: 9.4
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * 
 * @package WC_Quotes_Woodmart
 */

// Empêcher l'accès direct
if (!defined('ABSPATH')) {
    exit;
}

// Définir les constantes du plugin
define('WCQ_VERSION', '1.2.1');
define('WCQ_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('WCQ_PLUGIN_URL', plugin_dir_url(__FILE__));
define('WCQ_PLUGIN_BASENAME', plugin_basename(__FILE__));

/**
 * Classe principale du plugin
 */
class WC_Quotes_Woodmart {
    
    /**
     * Instance unique du plugin (Singleton)
     */
    private static $instance = null;
    
    /**
     * Obtenir l'instance unique du plugin
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructeur
     */
    private function __construct() {
        // Vérifier si WooCommerce est actif
        add_action('plugins_loaded', array($this, 'init'));
    }
    
    /**
     * Initialiser le plugin
     */
    public function init() {
        // Vérifier si WooCommerce est actif
        if (!class_exists('WooCommerce')) {
            add_action('admin_notices', array($this, 'woocommerce_missing_notice'));
            return;
        }
        
        // Déclarer la compatibilité avec les fonctionnalités WooCommerce
        add_action('before_woocommerce_init', array($this, 'declare_woocommerce_compatibility'));
        
        // Charger les traductions
        load_plugin_textdomain('wc-quotes-woodmart', false, dirname(WCQ_PLUGIN_BASENAME) . '/languages');
        
        // Inclure les fichiers nécessaires
        $this->includes();
        
        // Initialiser les hooks
        $this->init_hooks();
    }
    
    /**
     * Déclarer la compatibilité avec WooCommerce
     */
    public function declare_woocommerce_compatibility() {
        if (class_exists('\Automattic\WooCommerce\Utilities\FeaturesUtil')) {
            \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('custom_order_tables', __FILE__, true);
            \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('cart_checkout_blocks', __FILE__, true);
        }
    }
    
    /**
     * Inclure les fichiers du plugin
     */
    private function includes() {
        require_once WCQ_PLUGIN_DIR . 'includes/class-wcq-post-types.php';
        require_once WCQ_PLUGIN_DIR . 'includes/class-wcq-cart.php';
        require_once WCQ_PLUGIN_DIR . 'includes/class-wcq-quote.php';
        require_once WCQ_PLUGIN_DIR . 'includes/class-wcq-admin.php';
        require_once WCQ_PLUGIN_DIR . 'includes/class-wcq-customer.php';
        require_once WCQ_PLUGIN_DIR . 'includes/class-wcq-pdf.php';
        require_once WCQ_PLUGIN_DIR . 'includes/class-wcq-email.php';
    }
    
    /**
     * Initialiser les hooks
     */
    private function init_hooks() {
        // Hook d'activation
        register_activation_hook(__FILE__, array($this, 'activate'));
        
        // Hook de désactivation
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
        
        // Ajouter le template personnalisé
        add_filter('theme_page_templates', array($this, 'add_page_template'));
        add_filter('template_include', array($this, 'load_page_template'));
        
        // Ajouter un lien dans le menu admin
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_notices', array($this, 'check_edit_page_notice'));
        
        // Enregistrer les styles et scripts
        add_action('wp_enqueue_scripts', array($this, 'enqueue_frontend_scripts'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
    }
    
    /**
     * Activation du plugin
     */
    public function activate() {
        // Créer les tables nécessaires si besoin
        $this->create_tables();
        
        // Enregistrer l'endpoint WooCommerce
        add_option('woocommerce_myaccount_quotes_endpoint', 'quotes');
        
        // Créer la page d'édition de devis
        $this->create_edit_quote_page();
        
        // Charger les classes qui enregistrent les endpoints
        require_once WCQ_PLUGIN_DIR . 'includes/class-wcq-customer.php';
        
        // Déclencher l'enregistrement des endpoints
        $customer = new WCQ_Customer();
        $customer->add_quotes_endpoint();
        
        // Flush les règles de réécriture
        flush_rewrite_rules();
        
        // Marquer que le flush a été fait pour cette version
        update_option('wcq_rewrite_flushed', WCQ_VERSION);
        update_option('wcq_customer_rewrite_flushed', WCQ_VERSION);
    }
    
    /**
     * Désactivation du plugin
     */
    public function deactivate() {
        flush_rewrite_rules();
    }
    
    /**
     * Créer la page d'édition de devis
     */
    private function create_edit_quote_page() {
        // Vérifier si la page existe déjà et est valide
        $page_id = get_option('wcq_edit_quote_page_id');
        
        if ($page_id && get_post($page_id) && get_post_status($page_id) === 'publish') {
            return; // La page existe déjà et est publiée
        }
        
        // Chercher si une page avec ce titre existe déjà
        $existing_page = get_page_by_title('Modifier le devis', OBJECT, 'page');
        
        if ($existing_page) {
            // Utiliser la page existante
            $page_id = $existing_page->ID;
            update_post_meta($page_id, '_wp_page_template', 'edit-quote.php');
            update_option('wcq_edit_quote_page_id', $page_id);
            return;
        }
        
        // Créer la page
        $page_id = wp_insert_post(array(
            'post_title'     => 'Modifier le devis',
            'post_name'      => 'modifier-le-devis',
            'post_content'   => '<!-- Page gérée par WC Quotes Woodmart -->',
            'post_status'    => 'publish',
            'post_type'      => 'page',
            'post_author'    => 1,
            'comment_status' => 'closed',
            'ping_status'    => 'closed',
        ));
        
        if ($page_id && !is_wp_error($page_id)) {
            // Sauvegarder l'ID de la page dans les options
            update_option('wcq_edit_quote_page_id', $page_id, true);
            
            // Assigner le template
            update_post_meta($page_id, '_wp_page_template', 'edit-quote.php');
            
            // Logger pour debug
            error_log('WCQ: Page créée avec succès. ID: ' . $page_id);
        } else {
            error_log('WCQ: Erreur lors de la création de la page: ' . ($page_id instanceof WP_Error ? $page_id->get_error_message() : 'Unknown error'));
        }
    }
    
    /**
     * Ajouter le template à la liste des templates disponibles
     */
    public function add_page_template($templates) {
        $templates['edit-quote.php'] = __('Édition de Devis', 'wc-quotes-woodmart');
        return $templates;
    }
    
    /**
     * Charger le template personnalisé
     */
    public function load_page_template($template) {
        if (is_page()) {
            $page_template = get_post_meta(get_the_ID(), '_wp_page_template', true);
            
            if ($page_template === 'edit-quote.php') {
                $plugin_template = WCQ_PLUGIN_DIR . 'templates/edit-quote.php';
                
                if (file_exists($plugin_template)) {
                    return $plugin_template;
                }
            }
        }
        
        return $template;
    }
    
    /**
     * Ajouter un menu dans l'admin
     */
    public function add_admin_menu() {
        add_submenu_page(
            'woocommerce',
            __('Configuration Devis', 'wc-quotes-woodmart'),
            __('Configuration Devis', 'wc-quotes-woodmart'),
            'manage_woocommerce',
            'wcq-setup',
            array($this, 'render_setup_page')
        );
    }
    
    /**
     * Afficher un notice si la page n'existe pas
     */
    public function check_edit_page_notice() {
        $page_id = get_option('wcq_edit_quote_page_id');
        
        if (!$page_id || !get_post($page_id)) {
            ?>
            <div class="notice notice-warning is-dismissible">
                <p>
                    <strong>WC Quotes Woodmart:</strong> 
                    La page "Modifier le devis" n'est pas configurée. 
                    <a href="<?php echo admin_url('admin.php?page=wcq-setup'); ?>">Cliquez ici pour la créer automatiquement</a>
                </p>
            </div>
            <?php
        }
    }
    
    /**
     * Page de configuration
     */
    public function render_setup_page() {
        // Traiter les actions
        if (isset($_POST['wcq_create_page']) && check_admin_referer('wcq_setup')) {
            $this->create_edit_quote_page();
            echo '<div class="notice notice-success"><p>La page a été créée avec succès !</p></div>';
        }
        
        $page_id = get_option('wcq_edit_quote_page_id');
        $page_exists = $page_id && get_post($page_id);
        ?>
        <div class="wrap">
            <h1>Configuration WC Quotes Woodmart</h1>
            
            <div class="card">
                <h2>Page "Modifier le devis"</h2>
                
                <?php if ($page_exists) : 
                    $page = get_post($page_id);
                ?>
                    <p class="notice notice-success inline">
                        ✅ La page est configurée correctement
                    </p>
                    
                    <table class="form-table">
                        <tr>
                            <th>ID de la page:</th>
                            <td><code><?php echo $page_id; ?></code></td>
                        </tr>
                        <tr>
                            <th>Titre:</th>
                            <td><?php echo esc_html($page->post_title); ?></td>
                        </tr>
                        <tr>
                            <th>URL:</th>
                            <td><a href="<?php echo get_permalink($page_id); ?>" target="_blank"><?php echo get_permalink($page_id); ?></a></td>
                        </tr>
                        <tr>
                            <th>Statut:</th>
                            <td><?php echo esc_html($page->post_status); ?></td>
                        </tr>
                        <tr>
                            <th>Template:</th>
                            <td><code><?php echo esc_html(get_post_meta($page_id, '_wp_page_template', true)); ?></code></td>
                        </tr>
                    </table>
                    
                    <p>
                        <a href="<?php echo admin_url('post.php?post=' . $page_id . '&action=edit'); ?>" class="button">
                            Modifier la page
                        </a>
                    </p>
                    
                <?php else : ?>
                    
                    <p class="notice notice-warning inline">
                        ⚠️ La page "Modifier le devis" n'est pas configurée
                    </p>
                    
                    <p>
                        Cette page est nécessaire pour que les clients puissent modifier leurs devis.<br>
                        Sans cette page, le bouton "Modifier le devis" redirigera vers le panier normal.
                    </p>
                    
                    <form method="post">
                        <?php wp_nonce_field('wcq_setup'); ?>
                        <input type="hidden" name="wcq_create_page" value="1">
                        <p>
                            <button type="submit" class="button button-primary button-hero">
                                Créer la page automatiquement
                            </button>
                        </p>
                    </form>
                    
                    <hr>
                    
                    <h3>Ou créer manuellement :</h3>
                    <ol>
                        <li>Allez dans <strong>Pages → Ajouter</strong></li>
                        <li>Titre : <code>Modifier le devis</code></li>
                        <li>URL (permalien) : <code>modifier-le-devis</code></li>
                        <li>Contenu : Laissez vide</li>
                        <li>Template : Sélectionnez <strong>"Édition de Devis"</strong></li>
                        <li>Publiez la page</li>
                        <li>Revenez sur cette page, elle sera automatiquement détectée</li>
                    </ol>
                    
                <?php endif; ?>
            </div>
            
            <div class="card">
                <h2>Test de configuration</h2>
                
                <p>Pour tester que tout fonctionne :</p>
                <ol>
                    <li>Créez un devis depuis le panier</li>
                    <li>Allez dans "Mon compte → Mes devis"</li>
                    <li>Cliquez sur un devis</li>
                    <li>Cliquez sur "Modifier le devis"</li>
                    <li>Vous devez arriver sur : <code>/modifier-le-devis/?quote_id=...</code></li>
                </ol>
                
                <p>
                    <strong>Si vous arrivez sur <code>/panier/</code> à la place :</strong><br>
                    → La page n'est pas correctement configurée, utilisez le bouton ci-dessus pour la créer.
                </p>
            </div>
        </div>
        
        <style>
            .card { 
                background: white; 
                border: 1px solid #ccd0d4; 
                border-radius: 4px; 
                padding: 20px; 
                margin: 20px 0; 
                max-width: 800px;
            }
            .notice.inline { 
                padding: 10px 15px; 
                margin: 15px 0;
            }
        </style>
        <?php
    }
    
    /**
     * Créer les tables de base de données
     */
    private function create_tables() {
        global $wpdb;
        
        $charset_collate = $wpdb->get_charset_collate();
        $table_name = $wpdb->prefix . 'wc_quotes';
        
        $sql = "CREATE TABLE IF NOT EXISTS $table_name (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            quote_number varchar(50) NOT NULL,
            user_id bigint(20) NOT NULL,
            quote_data longtext NOT NULL,
            total decimal(10,2) NOT NULL,
            status varchar(20) NOT NULL DEFAULT 'pending',
            valid_until datetime DEFAULT NULL,
            created_at datetime NOT NULL,
            updated_at datetime NOT NULL,
            PRIMARY KEY (id),
            KEY user_id (user_id),
            KEY quote_number (quote_number)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }
    
    /**
     * Notice si WooCommerce n'est pas installé
     */
    public function woocommerce_missing_notice() {
        ?>
        <div class="error">
            <p><?php _e('WooCommerce Quotes for Woodmart nécessite WooCommerce pour fonctionner. Veuillez installer et activer WooCommerce.', 'wc-quotes-woodmart'); ?></p>
        </div>
        <?php
    }
    
    /**
     * Charger les scripts frontend
     */
    public function enqueue_frontend_scripts() {
        wp_enqueue_style(
            'wcq-frontend',
            WCQ_PLUGIN_URL . 'assets/css/frontend.css',
            array(),
            WCQ_VERSION
        );
        
        wp_enqueue_script(
            'wcq-frontend',
            WCQ_PLUGIN_URL . 'assets/js/frontend.js',
            array('jquery'),
            WCQ_VERSION,
            true
        );
        
        wp_localize_script('wcq-frontend', 'wcq_params', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('wcq_nonce'),
            'currency_symbol' => get_woocommerce_currency_symbol(),
            'currency_format' => str_replace(array('%1$s', '%2$s'), array('%1$s', '%2$s'), get_woocommerce_price_format()),
            'decimals' => wc_get_price_decimals(),
            'decimal_sep' => wc_get_price_decimal_separator(),
            'thousand_sep' => wc_get_price_thousand_separator(),
            'i18n' => array(
                'confirm_delete' => __('Êtes-vous sûr de vouloir supprimer ce devis ?', 'wc-quotes-woodmart'),
                'confirm_remove' => __('Êtes-vous sûr de vouloir supprimer cet article du devis ?', 'wc-quotes-woodmart'),
                'loading' => __('Chargement...', 'wc-quotes-woodmart')
            )
        ));
    }
    
    /**
     * Charger les scripts admin
     */
    public function enqueue_admin_scripts($hook) {
        // Charger uniquement sur les pages du plugin
        if (strpos($hook, 'wc-quotes') === false && get_post_type() !== 'shop_quote') {
            return;
        }
        
        wp_enqueue_style(
            'wcq-admin',
            WCQ_PLUGIN_URL . 'assets/css/admin.css',
            array(),
            WCQ_VERSION
        );
        
        wp_enqueue_script(
            'wcq-admin',
            WCQ_PLUGIN_URL . 'assets/js/admin.js',
            array('jquery'),
            WCQ_VERSION,
            true
        );
        
        wp_localize_script('wcq-admin', 'wcq_admin_params', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('wcq_admin_nonce'),
            'i18n' => array(
                'confirm_convert' => __('Êtes-vous sûr de vouloir convertir ce devis en commande ?', 'wc-quotes-woodmart'),
                'loading' => __('Chargement...', 'wc-quotes-woodmart')
            )
        ));
    }
}

/**
 * Fonction principale pour accéder à l'instance du plugin
 */
function WCQ() {
    return WC_Quotes_Woodmart::get_instance();
}

// Lancer le plugin
WCQ();
